import path from "path";
import "dotenv/config";
import * as express from "express";
import express__default from "express";
import cors from "cors";
import { createClient } from "@supabase/supabase-js";
import z, { z as z$1 } from "zod";
import bcrypt from "bcryptjs";
import nodemailer from "nodemailer";
const supabaseUrl = process.env.SUPABASE_URL;
const supabaseServiceKey = process.env.SUPABASE_SERVICE_ROLE_KEY;
if (!supabaseUrl || !supabaseServiceKey) {
  throw new Error("Supabase URL and Service Role Key must be defined in environment variables");
}
const supabase = createClient(supabaseUrl, supabaseServiceKey, {
  auth: {
    autoRefreshToken: false,
    persistSession: false,
    detectSessionInUrl: false
  },
  db: {
    schema: "public"
  }
});
(async () => {
  try {
    console.log("🧪 Testing Supabase connection...");
    const { data, error, count } = await supabase.from("users").select("id", { count: "exact", head: true });
    if (error) {
      console.error("❌ Supabase connection test FAILED:");
      console.error("Error:", error);
      throw error;
    }
    console.log("✅ Supabase connection test PASSED");
    console.log("✅ Users table accessible");
    console.log("✅ Total users in database:", count);
    console.log("✅ RLS bypassed successfully\n");
  } catch (error) {
    console.error("❌ CRITICAL: Cannot connect to Supabase");
    console.error("Error:", error);
    process.exit(1);
  }
})();
const SALT_ROUNDS = parseInt(process.env.BCRYPT_ROUNDS || "12");
class PasswordService {
  static async hashPassword(password) {
    try {
      const salt = await bcrypt.genSalt(SALT_ROUNDS);
      const hashedPassword = await bcrypt.hash(password, salt);
      return hashedPassword;
    } catch (error) {
      throw new Error("Failed to hash password");
    }
  }
  static async comparePassword(password, hashedPassword) {
    try {
      return await bcrypt.compare(password, hashedPassword);
    } catch (error) {
      throw new Error("Failed to compare password");
    }
  }
  static validatePasswordStrength(password) {
    const errors = [];
    let score = 0;
    if (password.length < 8) {
      errors.push("Password must be at least 8 characters long");
    } else {
      score += 1;
    }
    if (password.length > 128) {
      errors.push("Password must be less than 128 characters long");
    }
    if (!/[a-z]/.test(password)) {
      errors.push("Password must contain at least one lowercase letter");
    } else {
      score += 1;
    }
    if (!/[A-Z]/.test(password)) {
      errors.push("Password must contain at least one uppercase letter");
    } else {
      score += 1;
    }
    if (!/\d/.test(password)) {
      errors.push("Password must contain at least one number");
    } else {
      score += 1;
    }
    if (!/[!@#$%^&*(),.?":{}|<>]/.test(password)) {
      errors.push("Password must contain at least one special character");
    } else {
      score += 1;
    }
    const commonPatterns = [
      /123456/,
      /password/i,
      /qwerty/i,
      /admin/i,
      /letmein/i,
      /welcome/i,
      /monkey/i,
      /dragon/i
    ];
    if (commonPatterns.some((pattern) => pattern.test(password))) {
      errors.push("Password contains common patterns and is not secure");
      score = Math.max(0, score - 2);
    }
    if (/(.)\1{2,}/.test(password)) {
      errors.push("Password should not contain repeated characters");
      score = Math.max(0, score - 1);
    }
    return {
      isValid: errors.length === 0,
      errors,
      score: Math.min(5, score)
    };
  }
  static generateSecurePassword(length = 16) {
    const lowercase = "abcdefghijklmnopqrstuvwxyz";
    const uppercase = "ABCDEFGHIJKLMNOPQRSTUVWXYZ";
    const numbers = "0123456789";
    const symbols = "!@#$%^&*()_+-=[]{}|;:,.<>?";
    const allChars = lowercase + uppercase + numbers + symbols;
    let password = "";
    password += lowercase[Math.floor(Math.random() * lowercase.length)];
    password += uppercase[Math.floor(Math.random() * uppercase.length)];
    password += numbers[Math.floor(Math.random() * numbers.length)];
    password += symbols[Math.floor(Math.random() * symbols.length)];
    for (let i = 4; i < length; i++) {
      password += allChars[Math.floor(Math.random() * allChars.length)];
    }
    return password.split("").sort(() => Math.random() - 0.5).join("");
  }
}
const registerSchema = z.object({
  email: z.string().email("Invalid email format").toLowerCase().trim(),
  username: z.string().min(3, "Username must be at least 3 characters").max(20, "Username cannot exceed 20 characters").regex(/^[a-zA-Z0-9_]+$/, "Username can only contain letters, numbers, and underscores").optional(),
  firstName: z.string().min(1, "First name is required").max(50, "First name cannot exceed 50 characters").trim().optional(),
  lastName: z.string().min(1, "Last name is required").max(50, "Last name cannot exceed 50 characters").trim().optional(),
  password: z.string().min(8, "Password must be at least 8 characters").max(100, "Password cannot exceed 100 characters"),
  confirmPassword: z.string(),
  phone: z.string().regex(/^\+?[0-9]{10,15}$/, "Invalid phone number format").optional()
}).superRefine((data, ctx) => {
  if (data.password !== data.confirmPassword) {
    ctx.addIssue({
      code: z.ZodIssueCode.custom,
      message: "Passwords don't match",
      path: ["confirmPassword"]
    });
  }
});
const loginSchema = z.object({
  email: z.string().email("Invalid email format").toLowerCase().trim(),
  password: z.string().min(1, "Password is required"),
  rememberMe: z.boolean().optional().default(false)
});
const updateProfileSchema = z.object({
  firstName: z.string().min(1).max(50).trim().optional(),
  lastName: z.string().min(1).max(50).trim().optional(),
  username: z.string().min(3).max(20).regex(/^[a-zA-Z0-9_]+$/).optional(),
  phone: z.string().regex(/^\+?[0-9]{10,15}$/).optional(),
  email: z.string().email().toLowerCase().trim().optional()
});
const changePasswordSchema = z.object({
  currentPassword: z.string().min(1, "Current password is required"),
  newPassword: z.string().min(8, "Password must be at least 8 characters").max(100, "Password cannot exceed 100 characters"),
  confirmPassword: z.string()
}).superRefine((data, ctx) => {
  if (data.newPassword !== data.confirmPassword) {
    ctx.addIssue({
      code: z.ZodIssueCode.custom,
      message: "New passwords don't match",
      path: ["confirmPassword"]
    });
  }
});
const submitKYCSchema = z.object({
  idType: z.enum(["ssn", "drivers_license", "passport", "national_id", "ntin"], {
    errorMap: () => ({ message: "Invalid ID type" })
  }),
  idNumber: z.string().min(1, "ID number is required").max(50),
  frontImageUrl: z.string().url("Invalid front image URL"),
  backImageUrl: z.string().url("Invalid back image URL")
});
const passwordResetRequestSchema = z.object({
  email: z.string().email("Invalid email format").toLowerCase().trim()
});
const passwordResetSchema = z.object({
  token: z.string().min(1, "Reset token is required"),
  newPassword: z.string().min(8, "Password must be at least 8 characters").max(100, "Password cannot exceed 100 characters"),
  confirmPassword: z.string()
}).superRefine((data, ctx) => {
  if (data.newPassword !== data.confirmPassword) {
    ctx.addIssue({
      code: z.ZodIssueCode.custom,
      message: "Passwords don't match",
      path: ["confirmPassword"]
    });
  }
});
z.object({
  token_hash: z.string().min(1, "Verification token is required")
});
const resendVerificationSchema = z.object({
  email: z.string().email("Invalid email format"),
  emailRedirectTo: z.string()
});
const sanitizeUserData = (user) => {
  const { password, two_factor_secret, ...sanitized } = user;
  return sanitized;
};
const createSessionData = (session, user) => {
  return {
    access_token: session.access_token,
    refresh_token: session.refresh_token,
    expires_at: session.expires_at,
    expires_in: session.expires_in,
    token_type: session.token_type || "bearer",
    user: sanitizeUserData(user)
  };
};
const setCookies = (res, session, rememberMe = false) => {
  const cookieOptions = {
    httpOnly: true,
    secure: true,
    sameSite: "strict",
    path: "/"
  };
  res.cookie("sb-access-token", session.access_token, {
    ...cookieOptions,
    maxAge: rememberMe ? 30 * 24 * 60 * 60 * 1e3 : 24 * 60 * 60 * 1e3
    // 30 days or 1 day
  });
  res.cookie("sb-refresh-token", session.refresh_token, {
    ...cookieOptions,
    maxAge: rememberMe ? 365 * 24 * 60 * 60 * 1e3 : 7 * 24 * 60 * 60 * 1e3
    // 1 year or 7 days
  });
};
const clearCookies = (res) => {
  const cookieOptions = {
    httpOnly: true,
    secure: true,
    sameSite: "strict",
    path: "/"
  };
  res.clearCookie("sb-access-token", cookieOptions);
  res.clearCookie("sb-refresh-token", cookieOptions);
};
const registerUser = async (req, res) => {
  try {
    const validatedData = registerSchema.parse(req.body);
    const { emailRedirectTo } = req.body;
    const passwordValidation = PasswordService.validatePasswordStrength(validatedData.password);
    if (!passwordValidation.isValid) {
      return res.status(400).json({
        success: false,
        message: "Password does not meet security requirements",
        errors: passwordValidation.errors,
        code: "WEAK_PASSWORD"
      });
    }
    const existingChecks = [
      supabase.from("users").select("email").eq("email", validatedData.email).maybeSingle()
    ];
    if (validatedData.username) {
      existingChecks.push(
        supabase.from("users").select("username").eq("username", validatedData.username).maybeSingle()
      );
    }
    const existingResults = await Promise.all(existingChecks);
    if (existingResults[0].data) {
      return res.status(409).json({
        success: false,
        message: "Email already registered",
        code: "EMAIL_EXISTS"
      });
    }
    if (validatedData.username && existingResults[1]?.data) {
      return res.status(409).json({
        success: false,
        message: "Username already taken",
        code: "USERNAME_EXISTS"
      });
    }
    const { data: authData, error: authError } = await supabase.auth.signUp({
      email: validatedData.email,
      password: validatedData.password,
      options: {
        emailRedirectTo: emailRedirectTo || `${process.env.CLIENT_URL}/login`,
        data: {
          first_name: validatedData.firstName,
          last_name: validatedData.lastName,
          username: validatedData.username,
          phone: validatedData.phone,
          role: "USER"
        }
      }
    });
    if (authError || !authData.user) {
      console.error("Auth signup error:", authError);
      throw new Error(authError?.message || "Registration failed");
    }
    const profileData = {
      id: authData.user.id,
      email: validatedData.email,
      username: validatedData.username || null,
      first_name: validatedData.firstName || null,
      last_name: validatedData.lastName || null,
      phone: validatedData.phone || null,
      password: validatedData.password,
      role: "USER",
      status: "PENDING_VERIFICATION",
      email_verified: false,
      created_at: (/* @__PURE__ */ new Date()).toISOString(),
      updated_at: (/* @__PURE__ */ new Date()).toISOString()
    };
    const { error: profileError } = await supabase.from("users").insert(profileData);
    if (profileError) {
      console.error("Profile creation error:", profileError);
      await supabase.auth.admin.deleteUser(authData.user.id);
      throw new Error("Failed to create user profile");
    }
    return res.status(201).json({
      success: true,
      message: "Registration successful. Please check your email to verify your account.",
      needsEmailVerification: true,
      data: {
        user: null,
        profile: null,
        session: null
      }
    });
  } catch (error) {
    console.error("Registration error:", error);
    if (error instanceof z.ZodError) {
      return res.status(400).json({
        success: false,
        message: "Validation error",
        errors: error.errors.map((e) => ({ field: e.path.join("."), message: e.message })),
        code: "VALIDATION_ERROR"
      });
    }
    return res.status(500).json({
      success: false,
      message: error.message || "Registration failed",
      code: "REGISTRATION_FAILED"
    });
  }
};
const loginUser = async (req, res) => {
  try {
    const validatedData = loginSchema.parse(req.body);
    const { data: userData, error: userError } = await supabase.from("users").select("id, email, created_at, status, email_verified, role, first_name, last_name, username, phone").eq("email", validatedData.email).maybeSingle();
    if (userError) {
      console.error("User lookup error:", userError);
      return res.status(500).json({
        success: false,
        message: "Database error occurred",
        code: "DATABASE_ERROR"
      });
    }
    if (!userData) {
      return res.status(401).json({
        success: false,
        message: "Invalid credentials",
        code: "INVALID_CREDENTIALS"
      });
    }
    if (userData.status === "SUSPENDED") {
      return res.status(403).json({
        success: false,
        message: "Account suspended. Please contact support.",
        code: "ACCOUNT_SUSPENDED"
      });
    }
    if (userData.status === "DEACTIVATED") {
      return res.status(403).json({
        success: false,
        message: "Account deactivated. Please contact support to reactivate.",
        code: "ACCOUNT_DEACTIVATED"
      });
    }
    if (!userData.email_verified) {
      return res.status(403).json({
        success: false,
        message: "Email not verified. Please check your email and verify your account.",
        code: "EMAIL_NOT_VERIFIED"
      });
    }
    const { data: authData, error: authError } = await supabase.auth.signInWithPassword({
      email: validatedData.email,
      password: validatedData.password
    });
    if (authError || !authData.user || !authData.session) {
      console.error("Auth signin error:", authError);
      return res.status(401).json({
        success: false,
        message: "Invalid credentials",
        code: "INVALID_CREDENTIALS"
      });
    }
    setCookies(res, authData.session, validatedData.rememberMe);
    supabase.from("users").update({
      last_login_at: (/* @__PURE__ */ new Date()).toISOString(),
      updated_at: (/* @__PURE__ */ new Date()).toISOString()
    }).eq("id", authData.user.id).then(({ error }) => {
      if (error) console.warn("Failed to update last_login_at:", error);
    });
    const enrichedUser = {
      ...authData.user,
      role: userData.role,
      first_name: userData.first_name,
      last_name: userData.last_name,
      username: userData.username
    };
    return res.json({
      success: true,
      message: "Login successful",
      data: {
        user: sanitizeUserData(enrichedUser),
        profile: userData,
        session: createSessionData(authData.session, enrichedUser)
      }
    });
  } catch (error) {
    console.error("Login error:", error);
    if (error instanceof z.ZodError) {
      return res.status(400).json({
        success: false,
        message: "Validation error",
        errors: error.errors.map((e) => ({ field: e.path.join("."), message: e.message })),
        code: "VALIDATION_ERROR"
      });
    }
    return res.status(500).json({
      success: false,
      message: "Login failed. Please try again.",
      code: "LOGIN_FAILED"
    });
  }
};
const logoutUser = async (req, res) => {
  try {
    clearCookies(res);
    const token = req.cookies?.["sb-access-token"] || req.headers.authorization?.split(" ")[1];
    if (token) {
      const { error } = await supabase.auth.admin.signOut(token);
      if (error && error.message !== "User not found") {
        console.warn("Supabase signout warning:", error);
      }
    }
    return res.json({
      success: true,
      message: "Logged out successfully"
    });
  } catch (error) {
    console.error("Logout error:", error);
    return res.json({
      success: true,
      message: "Logged out successfully"
    });
  }
};
const getCurrentUser = async (req, res) => {
  try {
    const user = req.user;
    const { data: profile, error: profileError } = await supabase.from("users").select(`
        id, email, username, first_name, last_name, phone, role, status,
        email_verified, created_at, updated_at, last_login_at
      `).eq("id", user.id).maybeSingle();
    if (profileError) {
      console.error("Profile fetch error:", profileError);
      return res.status(500).json({
        success: false,
        message: "Failed to fetch user profile",
        code: "PROFILE_FETCH_FAILED"
      });
    }
    return res.json({
      success: true,
      message: "User retrieved successfully",
      data: {
        user: sanitizeUserData(user),
        profile: profile || null,
        session: null
        // Client handles session
      }
    });
  } catch (error) {
    console.error("getCurrentUser error:", error);
    return res.status(500).json({
      success: false,
      message: "Failed to retrieve user information",
      code: "USER_FETCH_FAILED"
    });
  }
};
const updateUserProfile = async (req, res) => {
  try {
    const validatedData = updateProfileSchema.parse(req.body);
    const userId = req.user.id;
    const updateData = {
      updated_at: (/* @__PURE__ */ new Date()).toISOString()
    };
    if (validatedData.firstName !== void 0) {
      updateData.first_name = validatedData.firstName;
    }
    if (validatedData.lastName !== void 0) {
      updateData.last_name = validatedData.lastName;
    }
    if (validatedData.phone !== void 0) {
      updateData.phone = validatedData.phone;
    }
    if (validatedData.username) {
      const { data: existingUser } = await supabase.from("users").select("id").eq("username", validatedData.username).neq("id", userId).maybeSingle();
      if (existingUser) {
        return res.status(409).json({
          success: false,
          message: "Username already taken",
          code: "USERNAME_EXISTS"
        });
      }
      updateData.username = validatedData.username;
    }
    if (validatedData.email) {
      const { data: existingEmail } = await supabase.from("users").select("id").eq("email", validatedData.email).neq("id", userId).maybeSingle();
      if (existingEmail) {
        return res.status(409).json({
          success: false,
          message: "Email already in use",
          code: "EMAIL_EXISTS"
        });
      }
      const token = req.cookies?.["sb-access-token"] || req.headers.authorization?.split(" ")[1];
      if (!token) {
        return res.status(401).json({
          success: false,
          message: "Session expired",
          code: "SESSION_EXPIRED"
        });
      }
      const { error: authError } = await supabase.auth.admin.updateUserById(
        userId,
        { email: validatedData.email }
      );
      if (authError) {
        console.error("Auth email update error:", authError);
        return res.status(500).json({
          success: false,
          message: "Failed to update email",
          code: "EMAIL_UPDATE_FAILED"
        });
      }
      updateData.email = validatedData.email;
      updateData.email_verified = false;
      updateData.email_verified_at = null;
    }
    const { data: updatedProfile, error } = await supabase.from("users").update(updateData).eq("id", userId).select().single();
    if (error) {
      console.error("Profile update error:", error);
      return res.status(500).json({
        success: false,
        message: "Failed to update profile",
        code: "PROFILE_UPDATE_FAILED"
      });
    }
    return res.json({
      success: true,
      message: "Profile updated successfully",
      data: sanitizeUserData(updatedProfile)
    });
  } catch (error) {
    console.error("Update profile error:", error);
    if (error instanceof z.ZodError) {
      return res.status(400).json({
        success: false,
        message: "Validation error",
        errors: error.errors.map((e) => ({
          field: e.path.join("."),
          message: e.message
        })),
        code: "VALIDATION_ERROR"
      });
    }
    return res.status(500).json({
      success: false,
      message: "Failed to update profile",
      code: "PROFILE_UPDATE_FAILED"
    });
  }
};
const changePassword = async (req, res) => {
  try {
    if (!req.user || !req.user.id || !req.user.email) {
      return res.status(401).json({
        success: false,
        message: "Authentication required",
        code: "UNAUTHORIZED"
      });
    }
    const validatedData = changePasswordSchema.parse(req.body);
    const user = req.user;
    const passwordValidation = PasswordService.validatePasswordStrength(
      validatedData.newPassword
    );
    if (!passwordValidation.isValid) {
      return res.status(400).json({
        success: false,
        message: "Password does not meet security requirements",
        errors: passwordValidation.errors,
        code: "WEAK_PASSWORD"
      });
    }
    if (validatedData.currentPassword === validatedData.newPassword) {
      return res.status(400).json({
        success: false,
        message: "New password must be different from current password",
        code: "SAME_PASSWORD"
      });
    }
    const { error: verifyError } = await supabase.auth.signInWithPassword({
      email: user.email,
      password: validatedData.currentPassword
    });
    if (verifyError) {
      return res.status(400).json({
        success: false,
        message: "Current password is incorrect",
        code: "INVALID_CURRENT_PASSWORD"
      });
    }
    const { error: updateError } = await supabase.auth.admin.updateUserById(
      user.id,
      { password: validatedData.newPassword }
    );
    if (updateError) {
      console.error("Password update error:", updateError);
      return res.status(500).json({
        success: false,
        message: "Failed to update password",
        code: "PASSWORD_UPDATE_FAILED"
      });
    }
    await supabase.from("users").update({
      password_changed_at: (/* @__PURE__ */ new Date()).toISOString(),
      updated_at: (/* @__PURE__ */ new Date()).toISOString()
    }).eq("id", user.id);
    supabase.from("audit_logs").insert({
      user_id: user.id,
      action: "password_changed",
      ip_address: req.ip,
      user_agent: req.headers["user-agent"],
      timestamp: (/* @__PURE__ */ new Date()).toISOString()
    }).then(({ error }) => {
      if (error) console.warn("Audit log failed:", error);
    });
    return res.json({
      success: true,
      message: "Password updated successfully"
    });
  } catch (error) {
    console.error("Change password error:", error);
    if (error instanceof z.ZodError) {
      return res.status(400).json({
        success: false,
        message: "Validation error",
        errors: error.errors.map((e) => ({
          field: e.path.join("."),
          message: e.message
        })),
        code: "VALIDATION_ERROR"
      });
    }
    return res.status(500).json({
      success: false,
      message: "Failed to change password",
      code: "PASSWORD_CHANGE_FAILED"
    });
  }
};
const submitKYC = async (req, res) => {
  try {
    const validatedData = submitKYCSchema.parse(req.body);
    const userId = req.user.id;
    const { data: currentKYC } = await supabase.from("users").select("id_verification_status, id_verified").eq("id", userId).single();
    if (currentKYC?.id_verified && currentKYC?.id_verification_status === "approved") {
      return res.status(400).json({
        success: false,
        message: "KYC already approved. Contact support for changes.",
        code: "KYC_ALREADY_APPROVED"
      });
    }
    if (currentKYC?.id_verification_status === "pending") {
      return res.status(400).json({
        success: false,
        message: "KYC documents are already under review",
        code: "KYC_PENDING"
      });
    }
    const { data: updatedUser, error } = await supabase.from("users").update({
      id_type: validatedData.idType,
      id_number: validatedData.idNumber,
      id_front_image_url: validatedData.frontImageUrl,
      id_back_image_url: validatedData.backImageUrl,
      id_verification_status: "pending",
      id_verified: false,
      id_submitted_at: (/* @__PURE__ */ new Date()).toISOString(),
      id_reviewed_at: null,
      updated_at: (/* @__PURE__ */ new Date()).toISOString()
    }).eq("id", userId).select().single();
    if (error) {
      console.error("KYC submission error:", error);
      return res.status(500).json({
        success: false,
        message: "Failed to submit KYC documents",
        code: "KYC_SUBMISSION_FAILED"
      });
    }
    supabase.from("audit_logs").insert({
      user_id: userId,
      action: "kyc_submitted",
      metadata: { id_type: validatedData.idType },
      ip_address: req.ip,
      user_agent: req.headers["user-agent"],
      timestamp: (/* @__PURE__ */ new Date()).toISOString()
    }).then(({ error: error2 }) => {
      if (error2) console.warn("Audit log failed:", error2);
    });
    return res.json({
      success: true,
      message: "KYC documents submitted successfully. You will be notified once reviewed.",
      data: {
        id_verification_status: updatedUser.id_verification_status,
        id_submitted_at: updatedUser.id_submitted_at
      }
    });
  } catch (error) {
    console.error("Submit KYC error:", error);
    if (error instanceof z.ZodError) {
      return res.status(400).json({
        success: false,
        message: "Validation error",
        errors: error.errors.map((e) => ({
          field: e.path.join("."),
          message: e.message
        })),
        code: "VALIDATION_ERROR"
      });
    }
    return res.status(500).json({
      success: false,
      message: "Failed to submit KYC documents",
      code: "KYC_SUBMISSION_FAILED"
    });
  }
};
const getKYCStatus = async (req, res) => {
  try {
    const userId = req.user.id;
    const { data: kycData, error } = await supabase.from("users").select(`
        id_type,
        id_number,
        id_verification_status,
        id_verified,
        id_submitted_at,
        id_reviewed_at,
        id_front_image_url,
        id_back_image_url
      `).eq("id", userId).single();
    if (error) {
      console.error("KYC status fetch error:", error);
      return res.status(500).json({
        success: false,
        message: "Failed to fetch KYC status",
        code: "KYC_FETCH_FAILED"
      });
    }
    return res.json({
      success: true,
      data: kycData
    });
  } catch (error) {
    console.error("Get KYC status error:", error);
    return res.status(500).json({
      success: false,
      message: "Failed to fetch KYC status",
      code: "KYC_FETCH_FAILED"
    });
  }
};
const deleteAccount = async (req, res) => {
  try {
    const userId = req.user.id;
    const { password, reason } = req.body;
    if (!password) {
      return res.status(400).json({
        success: false,
        message: "Password confirmation is required",
        code: "PASSWORD_REQUIRED"
      });
    }
    const { error: verifyError } = await supabase.auth.signInWithPassword({
      email: req.user.email,
      password
    });
    if (verifyError) {
      return res.status(400).json({
        success: false,
        message: "Incorrect password",
        code: "INVALID_PASSWORD"
      });
    }
    const { data: userData } = await supabase.from("users").select("current_balance, total_investment").eq("id", userId).single();
    if (userData && (userData.current_balance > 0 || userData.total_investment > 0)) {
      return res.status(400).json({
        success: false,
        message: "Cannot delete account with active investments. Please withdraw all funds first.",
        code: "ACTIVE_INVESTMENTS"
      });
    }
    const { error: updateError } = await supabase.from("users").update({
      status: "DEACTIVATED",
      updated_at: (/* @__PURE__ */ new Date()).toISOString()
    }).eq("id", userId);
    if (updateError) {
      console.error("Account deactivation error:", updateError);
      return res.status(500).json({
        success: false,
        message: "Failed to delete account",
        code: "DELETION_FAILED"
      });
    }
    supabase.from("audit_logs").insert({
      user_id: userId,
      action: "account_deleted",
      metadata: { reason: reason || "No reason provided" },
      ip_address: req.ip,
      user_agent: req.headers["user-agent"],
      timestamp: (/* @__PURE__ */ new Date()).toISOString()
    }).then(({ error }) => {
      if (error) console.warn("Audit log failed:", error);
    });
    await supabase.auth.admin.signOut(
      req.cookies?.["sb-access-token"] || req.headers.authorization?.split(" ")[1] || ""
    );
    res.clearCookie("sb-access-token");
    res.clearCookie("sb-refresh-token");
    return res.json({
      success: true,
      message: "Account deleted successfully"
    });
  } catch (error) {
    console.error("Delete account error:", error);
    return res.status(500).json({
      success: false,
      message: "Failed to delete account",
      code: "DELETION_FAILED"
    });
  }
};
const requestPasswordReset = async (req, res) => {
  try {
    const validatedData = passwordResetRequestSchema.parse(req.body);
    const { data: userData } = await supabase.from("users").select("email").eq("email", validatedData.email).maybeSingle();
    if (userData) {
      const { error } = await supabase.auth.resetPasswordForEmail(validatedData.email, {
        redirectTo: `${process.env.CLIENT_URL}/reset-password`
      });
      if (error) {
        console.error("Password reset email error:", error);
      }
    }
    return res.json({
      success: true,
      message: "If an account with this email exists, a password reset link has been sent."
    });
  } catch (error) {
    console.error("Password reset request error:", error);
    if (error instanceof z.ZodError) {
      return res.status(400).json({
        success: false,
        message: "Invalid email format",
        code: "VALIDATION_ERROR"
      });
    }
    return res.status(500).json({
      success: false,
      message: "Failed to process password reset request",
      code: "RESET_REQUEST_FAILED"
    });
  }
};
const resetPassword = async (req, res) => {
  try {
    const validatedData = passwordResetSchema.parse(req.body);
    const passwordValidation = PasswordService.validatePasswordStrength(validatedData.newPassword);
    if (!passwordValidation.isValid) {
      return res.status(400).json({
        success: false,
        message: "Password does not meet security requirements",
        errors: passwordValidation.errors,
        code: "WEAK_PASSWORD"
      });
    }
    try {
      const { data: { user }, error: tokenError } = await supabase.auth.getUser(validatedData.token);
      if (tokenError || !user) {
        console.error("Token verification error:", tokenError);
        return res.status(400).json({
          success: false,
          message: "Invalid or expired reset token",
          code: "INVALID_RESET_TOKEN"
        });
      }
      const supabaseWithToken = supabase.auth.admin;
      const { error } = await supabaseWithToken.updateUserById(user.id, {
        password: validatedData.newPassword
      });
      if (error) {
        console.error("Password update error:", error);
        return res.status(500).json({
          success: false,
          message: "Failed to update password",
          code: "PASSWORD_UPDATE_FAILED"
        });
      }
    } catch (updateError) {
      console.error("Password reset process error:", updateError);
      return res.status(400).json({
        success: false,
        message: "Password reset failed. Please request a new reset link.",
        code: "PASSWORD_RESET_FAILED"
      });
    }
    return res.json({
      success: true,
      message: "Password updated successfully. Please log in with your new password."
    });
  } catch (error) {
    console.error("Password reset error:", error);
    if (error instanceof z.ZodError) {
      return res.status(400).json({
        success: false,
        message: "Validation error",
        errors: error.errors.map((e) => ({ field: e.path.join("."), message: e.message })),
        code: "VALIDATION_ERROR"
      });
    }
    return res.status(500).json({
      success: false,
      message: "Password reset failed. Please try again.",
      code: "PASSWORD_RESET_FAILED"
    });
  }
};
const handleEmailConfirmation = async (req, res) => {
  try {
    const { access_token, refresh_token } = req.body;
    if (!access_token) {
      return res.status(400).json({
        success: false,
        message: "Access token is required",
        code: "MISSING_TOKEN"
      });
    }
    const { data: { user }, error: userError } = await supabase.auth.getUser(access_token);
    if (userError || !user) {
      return res.status(400).json({
        success: false,
        message: "Invalid or expired token",
        code: "INVALID_TOKEN"
      });
    }
    const { data: updatedUser, error: updateError } = await supabase.from("users").update({
      email_verified: true,
      status: "ACTIVE",
      email_verified_at: (/* @__PURE__ */ new Date()).toISOString(),
      updated_at: (/* @__PURE__ */ new Date()).toISOString()
    }).eq("id", user.id).select("*").single();
    if (updateError) {
      console.error("User update error:", updateError);
      return res.status(500).json({
        success: false,
        message: "Failed to update user verification status",
        code: "UPDATE_FAILED"
      });
    }
    const sessionData = {
      access_token,
      refresh_token,
      expires_at: Math.floor(Date.now() / 1e3) + 3600,
      // 1 hour from now
      expires_in: 3600,
      token_type: "bearer",
      user: sanitizeUserData(user)
    };
    return res.json({
      success: true,
      message: "Email verified successfully",
      data: {
        user: sanitizeUserData(user),
        profile: updatedUser,
        session: sessionData
      }
    });
  } catch (error) {
    console.error("Email confirmation error:", error);
    return res.status(500).json({
      success: false,
      message: error.message || "Email confirmation failed",
      code: "CONFIRMATION_FAILED"
    });
  }
};
const resendVerificationEmail = async (req, res) => {
  try {
    const { email, emailRedirectTo } = resendVerificationSchema.parse(req.body);
    const { data: user, error: userError } = await supabase.from("users").select("id, email_verified").eq("email", email).maybeSingle();
    if (userError || !user) {
      return res.status(404).json({
        success: false,
        message: "User not found",
        code: "USER_NOT_FOUND"
      });
    }
    if (user.email_verified) {
      return res.status(400).json({
        success: false,
        message: "Email already verified",
        code: "ALREADY_VERIFIED"
      });
    }
    const { error } = await supabase.auth.resend({
      type: "signup",
      email,
      options: {
        emailRedirectTo: emailRedirectTo || `${process.env.CLIENT_URL}/login`
      }
    });
    if (error) throw error;
    await supabase.from("verification_events").insert({
      user_id: user.id,
      event_type: "verification_resent",
      ip_address: req.ip,
      user_agent: req.headers["user-agent"]
    });
    return res.json({
      success: true,
      message: "Verification email resent successfully"
    });
  } catch (error) {
    console.error("Resend verification error:", error);
    if (error instanceof z.ZodError) {
      return res.status(400).json({
        success: false,
        message: "Validation error",
        errors: error.errors,
        code: "VALIDATION_ERROR"
      });
    }
    return res.status(500).json({
      success: false,
      message: error.message || "Failed to resend verification email",
      code: "RESEND_FAILED"
    });
  }
};
const authenticate = async (req, res, next) => {
  try {
    const token = req.cookies?.["sb-access-token"] || req.headers.authorization?.split(" ")[1];
    if (!token) {
      res.status(401).json({
        success: false,
        message: "Authentication required",
        code: "NO_TOKEN"
      });
      return;
    }
    const { data: { user }, error } = await supabase.auth.getUser(token);
    if (error || !user) {
      console.error("Auth verification error:", error);
      res.status(401).json({
        success: false,
        message: "Invalid or expired token",
        code: "INVALID_TOKEN"
      });
      return;
    }
    const { data: profile, error: profileError } = await supabase.from("users").select("status, role, email_verified, first_name, last_name").eq("id", user.id).maybeSingle();
    if (profileError) {
      console.error("Profile fetch error:", profileError);
      res.status(500).json({
        success: false,
        message: "Authentication service error",
        code: "AUTH_SERVICE_ERROR"
      });
      return;
    }
    if (profile?.status === "SUSPENDED") {
      res.status(403).json({
        success: false,
        message: "Account suspended",
        code: "ACCOUNT_SUSPENDED"
      });
      return;
    }
    if (profile?.status === "DEACTIVATED") {
      res.status(403).json({
        success: false,
        message: "Account deactivated",
        code: "ACCOUNT_DEACTIVATED"
      });
      return;
    }
    const enrichedUser = {
      id: user.id,
      email: user.email,
      ...user,
      role: profile?.role || "USER",
      status: profile?.status || "ACTIVE",
      email_verified: profile?.email_verified || false,
      first_name: profile?.first_name,
      last_name: profile?.last_name
    };
    req.user = enrichedUser;
    if (false) ;
    next();
  } catch (error) {
    console.error("Authentication middleware error:", error);
    res.status(500).json({
      success: false,
      message: "Authentication service error",
      code: "AUTH_SERVICE_ERROR"
    });
  }
};
const createRateLimiter = (windowMs = 15 * 60 * 1e3, maxRequests = 100, options = {}) => {
  const requests = /* @__PURE__ */ new Map();
  const cleanupInterval = setInterval(() => {
    const now = Date.now();
    const keysToDelete = [];
    for (const [key, entry] of requests.entries()) {
      if (now > entry.resetTime) {
        keysToDelete.push(key);
      }
    }
    keysToDelete.forEach((key) => requests.delete(key));
  }, Math.min(windowMs / 4, 6e4));
  const cleanup = () => {
    clearInterval(cleanupInterval);
    requests.clear();
  };
  process.on("SIGTERM", cleanup);
  process.on("SIGINT", cleanup);
  return (req, res, next) => {
    try {
      const identifier = options.keyGenerator ? options.keyGenerator(req) : getClientIdentifier(req);
      const now = Date.now();
      let requestInfo = requests.get(identifier);
      if (!requestInfo || now > requestInfo.resetTime) {
        requestInfo = {
          count: 1,
          resetTime: now + windowMs
        };
        requests.set(identifier, requestInfo);
      } else {
        requestInfo.count++;
      }
      const remaining = Math.max(0, maxRequests - requestInfo.count);
      const resetTime = Math.ceil((requestInfo.resetTime - now) / 1e3);
      res.set({
        "X-RateLimit-Limit": maxRequests.toString(),
        "X-RateLimit-Remaining": remaining.toString(),
        "X-RateLimit-Reset": new Date(requestInfo.resetTime).toISOString(),
        "X-RateLimit-Window": (windowMs / 1e3).toString()
      });
      if (requestInfo.count > maxRequests) {
        res.set("Retry-After", resetTime.toString());
        res.status(429).json({
          success: false,
          message: `Too many requests. Limit of ${maxRequests} requests per ${windowMs / 6e4} minutes exceeded.`,
          code: "RATE_LIMIT_EXCEEDED",
          retryAfter: resetTime,
          limit: maxRequests,
          remaining: 0,
          resetTime: new Date(requestInfo.resetTime).toISOString()
        });
        return;
      }
      next();
    } catch (error) {
      console.error("Rate limiter error:", error);
      {
        next();
      }
    }
  };
};
const getClientIdentifier = (req) => {
  const forwarded = req.headers["x-forwarded-for"];
  const realIP = req.headers["x-real-ip"];
  const clientIP = req.headers["x-client-ip"];
  let ip;
  if (typeof forwarded === "string") {
    ip = forwarded.split(",")[0].trim();
  } else if (typeof realIP === "string") {
    ip = realIP.trim();
  } else if (typeof clientIP === "string") {
    ip = clientIP.trim();
  } else {
    ip = req.ip || req.socket.remoteAddress || "unknown";
  }
  return ip;
};
const rateLimiters = {
  // Strict rate limiting for authentication endpoints
  auth: createRateLimiter(
    parseInt(process.env.AUTH_RATE_LIMIT_WINDOW_MS || "900000"),
    // 15 minutes
    parseInt(process.env.AUTH_RATE_LIMIT_MAX_REQUESTS || "5")
    // 5 requests
  ),
  // Moderate rate limiting for API endpoints
  api: createRateLimiter(
    parseInt(process.env.API_RATE_LIMIT_WINDOW_MS || "60000"),
    // 1 minute  
    parseInt(process.env.API_RATE_LIMIT_MAX_REQUESTS || "60")
    // 60 requests
  ),
  // Lenient rate limiting for public endpoints
  public: createRateLimiter(
    parseInt(process.env.PUBLIC_RATE_LIMIT_WINDOW_MS || "60000"),
    // 1 minute
    parseInt(process.env.PUBLIC_RATE_LIMIT_MAX_REQUESTS || "100")
    // 100 requests
  ),
  verification: createRateLimiter(
    parseInt(process.env.VERIFICATION_RATE_LIMIT_WINDOW_MS || "300000"),
    parseInt(process.env.VERIFICATION_RATE_LIMIT_MAX_REQUESTS || "1")
  )
};
const router$2 = express__default.Router();
router$2.post("/register", rateLimiters.auth, registerUser);
router$2.post("/login", rateLimiters.auth, loginUser);
router$2.post("/forgot-password", rateLimiters.auth, requestPasswordReset);
router$2.post("/reset-password", rateLimiters.auth, resetPassword);
router$2.post("/logout", authenticate, logoutUser);
router$2.get("/me", authenticate, getCurrentUser);
router$2.put("/profile", authenticate, rateLimiters.api, updateUserProfile);
router$2.post("/confirm-email", rateLimiters.verification, handleEmailConfirmation);
router$2.post("/resend-verification", rateLimiters.verification, resendVerificationEmail);
router$2.post("/profile/change-password", authenticate, rateLimiters.auth, changePassword);
router$2.post("/profile/kyc", rateLimiters.verification, submitKYC);
router$2.get("/profile/kyc", authenticate, getKYCStatus);
router$2.delete("/profile/account", rateLimiters.verification, deleteAccount);
class EmailService {
  transporter;
  adminEmails;
  constructor() {
    this.transporter = nodemailer.createTransport({
      host: process.env.SMTP_HOST,
      // e.g., mail.yourdomain.com
      port: parseInt(process.env.SMTP_PORT || "465"),
      // 465 for SSL, 587 for TLS
      secure: process.env.SMTP_PORT === "465",
      // true for 465, false for 587
      auth: {
        user: process.env.SMTP_USER,
        // e.g., notifications@yourdomain.com
        pass: process.env.SMTP_PASSWORD
      },
      // Important for cPanel hosting
      tls: {
        rejectUnauthorized: false
        // Useful for self-signed certificates on shared hosting
      }
    });
    this.adminEmails = process.env.ADMIN_EMAILS?.split(",").map((e) => e.trim()) || [];
  }
  /**
   * Send email using configured transporter
   */
  async sendEmail(options) {
    try {
      const info = await this.transporter.sendMail({
        from: `${process.env.SMTP_FROM_NAME || "Hyper Loop"} <${process.env.SMTP_FROM_EMAIL}>`,
        to: Array.isArray(options.to) ? options.to.join(", ") : options.to,
        subject: options.subject,
        html: options.html,
        text: options.text
      });
      console.log(`Email sent successfully: ${options.subject} - Message ID: ${info.messageId}`);
    } catch (error) {
      console.error("Failed to send email:", error);
      await this.logFailedEmail(options, error);
    }
  }
  /**
   * Log failed emails for retry
   */
  async logFailedEmail(options, error) {
    try {
      await supabase.from("failed_emails").insert({
        recipient: Array.isArray(options.to) ? options.to.join(",") : options.to,
        subject: options.subject,
        html: options.html,
        error_message: error.message,
        retry_count: 0,
        created_at: (/* @__PURE__ */ new Date()).toISOString()
      });
    } catch (dbError) {
      console.error("Failed to log email error:", dbError);
    }
  }
  /**
   * Generate base email template
   */
  getEmailTemplate(title, headerColor, content, buttonText, buttonUrl) {
    return `
      <!DOCTYPE html>
      <html>
        <head>
          <style>
            body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; margin: 0; padding: 0; }
            .container { max-width: 600px; margin: 0 auto; padding: 20px; }
            .header { background: ${headerColor}; color: white; padding: 20px; border-radius: 8px 8px 0 0; }
            .content { background: #f9fafb; padding: 30px; border: 1px solid #e5e7eb; }
            .detail-row { display: flex; justify-content: space-between; padding: 12px 0; border-bottom: 1px solid #e5e7eb; }
            .label { font-weight: 600; color: #6b7280; }
            .value { color: #111827; font-weight: 500; }
            .button { display: inline-block; background: #667eea; color: white; padding: 12px 24px; 
                      text-decoration: none; border-radius: 6px; margin-top: 20px; }
            .footer { text-align: center; padding: 20px; color: #6b7280; font-size: 12px; }
            .badge { display: inline-block; padding: 4px 12px; border-radius: 12px; font-size: 12px; font-weight: 600; }
            .badge-pending { background: #fef3c7; color: #92400e; }
            .alert { background: #fef2f2; border-left: 4px solid #dc2626; padding: 12px; margin: 20px 0; }
            .success { background: #f0fdf4; border-left: 4px solid #10b981; padding: 12px; margin: 20px 0; }
          </style>
        </head>
        <body>
          <div class="container">
            <div class="header">
              <h1 style="margin: 0;">${title}</h1>
            </div>
            <div class="content">
              ${content}
              ${buttonText && buttonUrl ? `
                <div style="text-align: center;">
                  <a href="${buttonUrl}" class="button">${buttonText}</a>
                </div>
              ` : ""}
            </div>
            <div class="footer">
              <p>This is an automated notification from Hyper Loop Admin System</p>
              <p>Please do not reply to this email</p>
            </div>
          </div>
        </body>
      </html>
    `;
  }
  /**
   * 1. Send deposit notification to admins
   */
  async sendDepositNotification(data) {
    if (this.adminEmails.length === 0) return;
    const content = `
      <div class="detail-row">
        <span class="label">Amount:</span>
        <span style="font-size: 24px; font-weight: bold; color: #10b981;">$${data.amount.toLocaleString()}</span>
      </div>
      <div class="detail-row">
        <span class="label">Method:</span>
        <span class="value">${data.method.replace("_", " ")}</span>
      </div>
      <div class="detail-row">
        <span class="label">Reference:</span>
        <span class="value">${data.reference}</span>
      </div>
      <div class="detail-row">
        <span class="label">Status:</span>
        <span class="badge badge-pending">PENDING</span>
      </div>
      <div class="detail-row">
        <span class="label">User:</span>
        <span class="value">${data.userName} (${data.userEmail})</span>
      </div>
      <div class="detail-row">
        <span class="label">Date:</span>
        <span class="value">${(/* @__PURE__ */ new Date()).toLocaleString()}</span>
      </div>
    `;
    const html = this.getEmailTemplate(
      "New Deposit Request",
      "linear-gradient(135deg, #667eea 0%, #764ba2 100%)",
      content,
      "View Deposit Details",
      `${process.env.ADMIN_DASHBOARD_URL}/deposits/${data.depositId}`
    );
    await this.sendEmail({
      to: this.adminEmails,
      subject: `New Deposit Request - $${data.amount.toLocaleString()}`,
      html
    });
    await this.logNotification("DEPOSIT", data.depositId, this.adminEmails);
  }
  /**
   * 2. Send withdrawal notification to admins
   */
  async sendWithdrawalNotification(data) {
    if (this.adminEmails.length === 0) return;
    const content = `
      <div class="alert">
        <strong>Important:</strong> Please verify user balance and approve/reject this withdrawal request.
      </div>
      <div class="detail-row">
        <span class="label">Amount:</span>
        <span style="font-size: 24px; font-weight: bold; color: #dc2626;">$${data.amount.toLocaleString()}</span>
      </div>
      <div class="detail-row">
        <span class="label">Method:</span>
        <span class="value">${data.method.replace("_", " ")}</span>
      </div>
      <div class="detail-row">
        <span class="label">Reference:</span>
        <span class="value">${data.reference}</span>
      </div>
      <div class="detail-row">
        <span class="label">User:</span>
        <span class="value">${data.userName} (${data.userEmail})</span>
      </div>
      <div class="detail-row">
        <span class="label">Date:</span>
        <span class="value">${(/* @__PURE__ */ new Date()).toLocaleString()}</span>
      </div>
    `;
    const html = this.getEmailTemplate(
      "New Withdrawal Request",
      "linear-gradient(135deg, #f59e0b 0%, #dc2626 100%)",
      content,
      "Review & Process",
      `${process.env.ADMIN_DASHBOARD_URL}/withdrawals/${data.withdrawalId}`
    );
    await this.sendEmail({
      to: this.adminEmails,
      subject: `URGENT: Withdrawal Request - $${data.amount.toLocaleString()}`,
      html
    });
    await this.logNotification("WITHDRAWAL", data.withdrawalId, this.adminEmails);
  }
  /**
   * 3. Send new investment notification
   */
  async sendInvestmentNotification(data) {
    if (this.adminEmails.length === 0) return;
    const content = `
      <div class="success">
        New investment created by user
      </div>
      <div class="detail-row">
        <span class="label">Investment Name:</span>
        <span class="value">${data.investmentName}</span>
      </div>
      <div class="detail-row">
        <span class="label">Amount:</span>
        <span style="font-size: 20px; font-weight: bold; color: #10b981;">$${data.amount.toLocaleString()}</span>
      </div>
      ${data.packageName ? `
      <div class="detail-row">
        <span class="label">Package:</span>
        <span class="value">${data.packageName}</span>
      </div>
      ` : ""}
      <div class="detail-row">
        <span class="label">Duration:</span>
        <span class="value">${data.duration} days</span>
      </div>
      <div class="detail-row">
        <span class="label">Risk Level:</span>
        <span class="value">${data.riskLevel}</span>
      </div>
      <div class="detail-row">
        <span class="label">User:</span>
        <span class="value">${data.userName} (${data.userEmail})</span>
      </div>
      <div class="detail-row">
        <span class="label">Date:</span>
        <span class="value">${(/* @__PURE__ */ new Date()).toLocaleString()}</span>
      </div>
    `;
    const html = this.getEmailTemplate(
      "New Investment Created",
      "linear-gradient(135deg, #10b981 0%, #059669 100%)",
      content,
      "View Investment",
      `${process.env.ADMIN_DASHBOARD_URL}/investments/${data.investmentId}`
    );
    await this.sendEmail({
      to: this.adminEmails,
      subject: `New Investment - $${data.amount.toLocaleString()} by ${data.userName}`,
      html
    });
    await this.logNotification("INVESTMENT", data.investmentId, this.adminEmails);
  }
  /**
   * 4. Send KYC submission notification
   */
  async sendKYCNotification(data) {
    if (this.adminEmails.length === 0) return;
    const content = `
      <div class="alert">
        <strong>Action Required:</strong> New KYC documents submitted for verification.
      </div>
      <div class="detail-row">
        <span class="label">User:</span>
        <span class="value">${data.userName} (${data.userEmail})</span>
      </div>
      <div class="detail-row">
        <span class="label">User ID:</span>
        <span class="value">${data.userId}</span>
      </div>
      <div class="detail-row">
        <span class="label">Documents Submitted:</span>
        <span class="value">${data.documentTypes.join(", ")}</span>
      </div>
      <div class="detail-row">
        <span class="label">Submitted At:</span>
        <span class="value">${new Date(data.submittedAt).toLocaleString()}</span>
      </div>
      <div style="margin-top: 20px; padding: 15px; background: #fffbeb; border-radius: 6px;">
        <p style="margin: 0; color: #92400e;">
          <strong>Next Steps:</strong> Review the submitted documents and approve or reject the KYC application.
        </p>
      </div>
    `;
    const html = this.getEmailTemplate(
      "New KYC Submission",
      "linear-gradient(135deg, #3b82f6 0%, #1d4ed8 100%)",
      content,
      "Review KYC Documents",
      `${process.env.ADMIN_DASHBOARD_URL}/kyc/${data.kycId}`
    );
    await this.sendEmail({
      to: this.adminEmails,
      subject: `KYC Verification Required - ${data.userName}`,
      html
    });
    await this.logNotification("KYC", data.kycId, this.adminEmails);
  }
  /**
   * 5. Send profile update notification (for sensitive changes)
   */
  async sendProfileUpdateNotification(data) {
    if (this.adminEmails.length === 0) return;
    const content = `
      <div style="background: #eff6ff; padding: 15px; border-radius: 6px; margin-bottom: 20px;">
        <p style="margin: 0; color: #1e40af;">
          User has updated their profile information.
        </p>
      </div>
      <div class="detail-row">
        <span class="label">User:</span>
        <span class="value">${data.userName} (${data.userEmail})</span>
      </div>
      <div class="detail-row">
        <span class="label">Fields Changed:</span>
        <span class="value">${data.changedFields.join(", ")}</span>
      </div>
      <div class="detail-row">
        <span class="label">Updated At:</span>
        <span class="value">${new Date(data.updatedAt).toLocaleString()}</span>
      </div>
    `;
    const html = this.getEmailTemplate(
      "Profile Update Notification",
      "linear-gradient(135deg, #8b5cf6 0%, #6d28d9 100%)",
      content,
      "View User Profile",
      `${process.env.ADMIN_DASHBOARD_URL}/users/${data.userId}`
    );
    await this.sendEmail({
      to: this.adminEmails,
      subject: `Profile Updated - ${data.userName}`,
      html
    });
    await this.logNotification("PROFILE_UPDATE", data.userId, this.adminEmails);
  }
  /**
   * 6. Send password change notification (security alert)
   */
  async sendPasswordChangeNotification(data) {
    if (this.adminEmails.length === 0) return;
    const content = `
      <div class="alert">
        <strong>Security Alert:</strong> User password has been changed.
      </div>
      <div class="detail-row">
        <span class="label">User:</span>
        <span class="value">${data.userName} (${data.userEmail})</span>
      </div>
      <div class="detail-row">
        <span class="label">Changed At:</span>
        <span class="value">${new Date(data.changedAt).toLocaleString()}</span>
      </div>
      ${data.ipAddress ? `
      <div class="detail-row">
        <span class="label">IP Address:</span>
        <span class="value">${data.ipAddress}</span>
      </div>
      ` : ""}
      <div style="margin-top: 20px; padding: 15px; background: #fef2f2; border-radius: 6px;">
        <p style="margin: 0; color: #991b1b;">
          <strong>Security Note:</strong> Monitor for any suspicious activity on this account.
        </p>
      </div>
    `;
    const html = this.getEmailTemplate(
      "Password Changed",
      "linear-gradient(135deg, #ef4444 0%, #b91c1c 100%)",
      content,
      "View User Activity",
      `${process.env.ADMIN_DASHBOARD_URL}/users/${data.userId}/activity`
    );
    await this.sendEmail({
      to: this.adminEmails,
      subject: `Security Alert: Password Changed - ${data.userName}`,
      html
    });
    await this.logNotification("PASSWORD_CHANGE", data.userId, this.adminEmails);
  }
  /**
   * 7. Send large transaction notification (threshold-based alert)
   */
  async sendLargeTransactionNotification(data) {
    if (this.adminEmails.length === 0) return;
    const content = `
      <div class="alert">
        <strong>Large Transaction Alert:</strong> Transaction exceeds threshold amount.
      </div>
      <div class="detail-row">
        <span class="label">Transaction Type:</span>
        <span class="value">${data.transactionType}</span>
      </div>
      <div class="detail-row">
        <span class="label">Amount:</span>
        <span style="font-size: 24px; font-weight: bold; color: #dc2626;">$${data.amount.toLocaleString()}</span>
      </div>
      <div class="detail-row">
        <span class="label">User:</span>
        <span class="value">${data.userName} (${data.userEmail})</span>
      </div>
      <div class="detail-row">
        <span class="label">Date:</span>
        <span class="value">${(/* @__PURE__ */ new Date()).toLocaleString()}</span>
      </div>
      <div style="margin-top: 20px; padding: 15px; background: #fffbeb; border-radius: 6px;">
        <p style="margin: 0; color: #92400e;">
          <strong>Action:</strong> Review this transaction for compliance and fraud prevention.
        </p>
      </div>
    `;
    const html = this.getEmailTemplate(
      "Large Transaction Alert",
      "linear-gradient(135deg, #f59e0b 0%, #d97706 100%)",
      content,
      "Review Transaction",
      `${process.env.ADMIN_DASHBOARD_URL}/transactions/${data.transactionId}`
    );
    await this.sendEmail({
      to: this.adminEmails,
      subject: `ALERT: Large Transaction - $${data.amount.toLocaleString()}`,
      html
    });
    await this.logNotification("LARGE_TRANSACTION", data.transactionId, this.adminEmails);
  }
  /**
   * Log notification to database for audit trail
   */
  async logNotification(type, relatedId, recipients) {
    try {
      await supabase.from("admin_notifications").insert({
        type,
        related_id: relatedId,
        recipients: recipients.join(","),
        sent_at: (/* @__PURE__ */ new Date()).toISOString(),
        status: "SENT"
      });
    } catch (error) {
      console.error("Failed to log notification:", error);
    }
  }
  /**
   * Test email configuration
   */
  async testConnection() {
    try {
      await this.transporter.verify();
      console.log("Email server connection verified");
      return true;
    } catch (error) {
      console.error("Email server connection failed:", error);
      return false;
    }
  }
  /**
   * Send test email to verify setup
   */
  async sendTestEmail(recipient) {
    const html = this.getEmailTemplate(
      "Test Email - System Working",
      "linear-gradient(135deg, #10b981 0%, #059669 100%)",
      `
        <div class="success">
          Your email notification system is configured correctly!
        </div>
        <div class="detail-row">
          <span class="label">Test Time:</span>
          <span class="value">${(/* @__PURE__ */ new Date()).toLocaleString()}</span>
        </div>
        <div class="detail-row">
          <span class="label">SMTP Host:</span>
          <span class="value">${process.env.SMTP_HOST}</span>
        </div>
        <div class="detail-row">
          <span class="label">From Email:</span>
          <span class="value">${process.env.SMTP_FROM_EMAIL}</span>
        </div>
      `
    );
    await this.sendEmail({
      to: recipient,
      subject: "Test Email - Hyper Loop Notifications",
      html
    });
  }
}
const emailService = new EmailService();
const createPortfolioSchema = z$1.object({
  name: z$1.string().min(1, "Portfolio name is required"),
  description: z$1.string().optional()
});
const createInvestmentSchema = z$1.object({
  packageId: z$1.string().optional(),
  name: z$1.string().min(1, "Investment name is required"),
  description: z$1.string().optional(),
  amount: z$1.number().positive("Investment amount must be positive"),
  duration: z$1.number().positive("Duration must be positive"),
  riskLevel: z$1.enum(["LOW", "MODERATE", "HIGH", "AGGRESSIVE"]).optional()
});
const createWithdrawalSchema = z$1.object({
  amount: z$1.number().positive("Withdrawal amount must be positive"),
  method: z$1.enum(["BANK_TRANSFER", "CRYPTO", "PAYPAL", "CHECK"]),
  bankAccount: z$1.string().optional(),
  cryptoAddress: z$1.string().optional(),
  notes: z$1.string().optional()
});
const createDepositSchema = z$1.object({
  amount: z$1.number().positive("Deposit amount must be positive").min(10, "Minimum deposit amount is $10"),
  method: z$1.enum(["BANK_TRANSFER", "CRYPTO", "PAYPAL"]),
  paypalEmail: z$1.string().email().optional(),
  note: z$1.string().max(500, "Notes cannot exceed 500 characters").optional(),
  cryptoType: z$1.enum(["btc", "eth"]).optional()
});
const updateDepositStatusSchema = z$1.object({
  status: z$1.enum(["PENDING", "PROCESSING", "COMPLETED", "FAILED", "CANCELLED"]),
  adminNotes: z$1.string().optional(),
  transactionHash: z$1.string().optional(),
  failureReason: z$1.string().optional()
});
class DashboardController {
  // Get user dashboard overview
  static async getDashboardOverview(req, res) {
    try {
      if (!req.user) {
        return DashboardController.sendError(res, 401, "Authentication required", "AUTH_REQUIRED");
      }
      const userId = req.user.id;
      const { data: user, error: userError } = await supabase.from("users").select(
        `id, email, first_name, last_name, 
          total_investment, current_balance, total_returns, 
          created_at, last_login_at`
      ).eq("id", userId).single();
      if (userError || !user) {
        return DashboardController.sendError(res, 404, "User not found", "USER_NOT_FOUND");
      }
      const [
        portfolios,
        recentTransactions,
        activeInvestments,
        pendingWithdrawals,
        monthlyPerformance,
        riskDistribution
      ] = await Promise.all([
        DashboardController.getUserPortfolios(userId),
        DashboardController.getRecentTransactions(userId),
        DashboardController.getActiveInvestments(userId),
        DashboardController.getPendingWithdrawals(userId),
        DashboardController.getMonthlyPerformance(userId),
        DashboardController.getRiskDistribution(userId)
      ]);
      const totalInvestment = Number(user.total_investment || 0);
      const currentBalance = Number(user.current_balance || 0);
      const totalReturns = Number(user.total_returns || 0);
      const performancePercent = totalInvestment > 0 ? totalReturns / totalInvestment * 100 : 0;
      res.json({
        success: true,
        data: {
          user: {
            id: user.id,
            email: user.email,
            name: `${user.first_name || ""} ${user.last_name || ""}`.trim(),
            joinedAt: user.created_at,
            lastLoginAt: user.last_login_at
          },
          summary: {
            totalInvestment,
            currentBalance,
            totalReturns,
            performancePercent: Number(performancePercent.toFixed(2)),
            portfolioCount: portfolios.length,
            activeInvestments: activeInvestments.length,
            pendingWithdrawals: pendingWithdrawals.length
          },
          portfolios,
          recentTransactions,
          activeInvestments,
          pendingWithdrawals,
          charts: {
            monthlyPerformance,
            riskDistribution
          }
        }
      });
    } catch (error) {
      console.error("Dashboard overview error:", error);
      DashboardController.sendError(
        res,
        500,
        "Failed to get dashboard overview",
        "DASHBOARD_ERROR"
      );
    }
  }
  // Get user portfolios
  static async getPortfolios(req, res) {
    try {
      if (!req.user) {
        return DashboardController.sendError(res, 401, "Authentication required", "AUTH_REQUIRED");
      }
      const portfolios = await DashboardController.getUserPortfolios(req.user.id, true);
      res.json({
        success: true,
        data: { portfolios }
      });
    } catch (error) {
      console.error("Get portfolios error:", error);
      DashboardController.sendError(res, 500, "Failed to get portfolios", "PORTFOLIOS_ERROR");
    }
  }
  // Create new portfolio
  static async createPortfolio(req, res) {
    try {
      if (!req.user) {
        return DashboardController.sendError(res, 401, "Authentication required", "AUTH_REQUIRED");
      }
      const validatedData = createPortfolioSchema.parse(req.body);
      const { count: existingPortfoliosCount } = await supabase.from("portfolios").select("*", { count: "exact", head: true }).eq("id", req.user.id);
      const { data: portfolio, error } = await supabase.from("portfolios").insert({
        id: req.user.id,
        name: validatedData.name,
        description: validatedData.description,
        is_default: existingPortfoliosCount === 0,
        total_value: 0,
        total_invested: 0,
        total_returns: 0,
        performance: 0
      }).select().single();
      if (error) throw error;
      res.status(201).json({
        success: true,
        message: "Portfolio created successfully",
        data: { portfolio }
      });
    } catch (error) {
      console.error("Create portfolio error:", error);
      if (error instanceof z$1.ZodError) {
        return DashboardController.sendValidationError(res, error);
      }
      DashboardController.sendError(
        res,
        500,
        "Failed to create portfolio",
        "CREATE_PORTFOLIO_ERROR"
      );
    }
  }
  // Get Packages
  static async getPackages(req, res) {
    try {
      const query = supabase.from("packages").select("*").order("created_at");
      const { data: packages, error } = await query;
      if (error) throw error;
      res.json({
        success: true,
        data: packages || []
      });
      return;
    } catch (error) {
      console.error("Get packages error:", error);
      DashboardController.sendError(
        res,
        500,
        "Failed to fetch packages",
        "GET_PACKAGES_ERROR"
      );
      return;
    }
  }
  // Get user investments
  static async getInvestments(req, res) {
    try {
      if (!req.user) {
        return DashboardController.sendError(res, 401, "Authentication required", "AUTH_REQUIRED");
      }
      const { status, page = 1, limit = 20 } = req.query;
      const offset = (Number(page) - 1) * Number(limit);
      const query = supabase.from("investments").select(
        `*, 
          package:packages(name, category)`,
        { count: "exact" }
      ).eq("user_id", req.user.id).order("created_at", { ascending: false }).range(offset, offset + Number(limit) - 1);
      if (status) query.eq("status", status);
      const { data: investments, error, count } = await query;
      if (error) throw error;
      res.json({
        success: true,
        data: {
          investments: investments?.map((i) => ({
            ...i,
            amount: Number(i.amount),
            currentValue: Number(i.current_value),
            returns: Number(i.returns),
            returnPercent: Number(i.return_percent)
          })) || [],
          pagination: {
            page: Number(page),
            limit: Number(limit),
            total: count || 0,
            pages: Math.ceil((count || 0) / Number(limit))
          }
        }
      });
    } catch (error) {
      console.error("Get investments error:", error);
      DashboardController.sendError(res, 500, "Failed to get investments", "INVESTMENTS_ERROR");
    }
  }
  // Create new investment
  static async createInvestment(req, res) {
    try {
      if (!req.user) {
        return DashboardController.sendError(res, 401, "Authentication required", "AUTH_REQUIRED");
      }
      const validatedData = createInvestmentSchema.parse(req.body);
      const { data: user, error: userError } = await supabase.from("users").select("current_balance").eq("id", req.user.id).single();
      if (userError || !user || Number(user.current_balance) < validatedData.amount) {
        return DashboardController.sendError(res, 400, "Insufficient balance", "INSUFFICIENT_BALANCE");
      }
      const startDate = (/* @__PURE__ */ new Date()).toISOString();
      const endDate = new Date(
        (/* @__PURE__ */ new Date()).getTime() + validatedData.duration * 24 * 60 * 60 * 1e3
      ).toISOString();
      const { data: investment, error } = await supabase.rpc("create_investment", {
        user_id: req.user.id,
        package_id: validatedData.packageId,
        name: validatedData.name,
        description: validatedData.description,
        amount: validatedData.amount,
        duration: validatedData.duration,
        start_date: startDate,
        end_date: endDate,
        risk_level: validatedData.riskLevel || "MODERATE"
      });
      if (error) throw error;
      res.status(201).json({
        success: true,
        message: "Investment created successfully",
        data: { investment }
      });
      emailService.sendInvestmentNotification({
        userId: req.user.id,
        userEmail: req.user.email || "",
        userName: validatedData.name,
        amount: validatedData.amount,
        packageName: validatedData.name,
        investmentName: validatedData.name,
        duration: validatedData.duration,
        riskLevel: validatedData.riskLevel,
        investmentId: investment.id
      }).catch(console.error);
    } catch (error) {
      console.error("Create investment error:", error);
      if (error instanceof z$1.ZodError) {
        return DashboardController.sendValidationError(res, error);
      }
      DashboardController.sendError(
        res,
        500,
        "Failed to create investment",
        "CREATE_INVESTMENT_ERROR"
      );
    }
  }
  // Get user transactions
  static async getTransactions(req, res) {
    try {
      if (!req.user) {
        return DashboardController.sendError(res, 401, "Authentication required", "AUTH_REQUIRED");
      }
      const { type, status, page = 1, limit = 20 } = req.query;
      const offset = (Number(page) - 1) * Number(limit);
      const query = supabase.from("transactions").select("*", { count: "exact" }).eq("id", req.user.id).order("created_at", { ascending: false }).range(offset, offset + Number(limit) - 1);
      if (type) query.eq("type", type);
      if (status) query.eq("status", status);
      const { data: transactions, error, count } = await query;
      if (error) throw error;
      res.json({
        success: true,
        data: {
          transactions: transactions?.map((t) => ({
            ...t,
            amount: Number(t.amount),
            fee: Number(t.fee || 0),
            quantity: t.quantity ? Number(t.quantity) : null,
            price: t.price ? Number(t.price) : null
          })) || [],
          pagination: {
            page: Number(page),
            limit: Number(limit),
            total: count || 0,
            pages: Math.ceil((count || 0) / Number(limit))
          }
        }
      });
    } catch (error) {
      console.error("Get transactions error:", error);
      DashboardController.sendError(res, 500, "Failed to get transactions", "TRANSACTIONS_ERROR");
    }
  }
  /**
   * Get deposit configuration details (bank, crypto, PayPal)
   * This fetches the system-wide deposit configuration
   */
  static async getDepositDetails(req, res) {
    try {
      if (!req.user) {
        return DashboardController.sendError(res, 401, "Authentication required", "AUTH_REQUIRED");
      }
      const { data: depositData, error: depositError } = await supabase.from("deposit_details").select("*").order("created_at", { ascending: false }).limit(1).single();
      if (depositError) {
        console.error("Error fetching deposit details:", depositError);
        return DashboardController.sendError(
          res,
          500,
          "Failed to fetch deposit details",
          "DEPOSIT_DETAILS_ERROR"
        );
      }
      if (!depositData) {
        return DashboardController.sendError(
          res,
          404,
          "No deposit details found",
          "DEPOSIT_DETAILS_NOT_FOUND"
        );
      }
      const { data: cryptoData, error: cryptoError } = await supabase.from("crypto_addresses").select("*").eq("deposit_details_id", depositData.id);
      if (cryptoError) {
        console.error("Error fetching crypto addresses:", cryptoError);
        return DashboardController.sendError(
          res,
          500,
          "Failed to fetch crypto addresses",
          "CRYPTO_ADDRESSES_ERROR"
        );
      }
      const cryptoAddresses = {};
      if (cryptoData && cryptoData.length > 0) {
        cryptoData.forEach((crypto) => {
          cryptoAddresses[crypto.crypto_type] = {
            address: crypto.address,
            minConfirmations: crypto.min_confirmations,
            network: crypto.network
          };
        });
      }
      res.json({
        success: true,
        data: {
          bank: {
            accountName: depositData.bank_account_name,
            accountNumber: depositData.bank_account_number,
            routingNumber: depositData.bank_routing_number,
            swiftCode: depositData.bank_swift_code
          },
          crypto: cryptoAddresses,
          paypal: {
            merchantEmail: depositData.paypal_merchant_email,
            merchantId: depositData.paypal_merchant_id,
            returnUrl: depositData.paypal_return_url,
            cancelUrl: depositData.paypal_cancel_url
          }
        }
      });
    } catch (error) {
      console.error("Error in getDepositDetails:", error);
      DashboardController.sendError(
        res,
        500,
        "Internal server error",
        "INTERNAL_ERROR"
      );
    }
  }
  /**
   * Create a deposit request
   */
  static async createDeposit(req, res) {
    try {
      if (!req.user) {
        return DashboardController.sendError(res, 401, "Authentication required", "AUTH_REQUIRED");
      }
      const validatedData = createDepositSchema.parse(req.body);
      if (validatedData.method === "PAYPAL" && !validatedData.paypalEmail) {
        return DashboardController.sendError(
          res,
          400,
          "PayPal email is required for PayPal deposits",
          "VALIDATION_ERROR"
        );
      }
      if (validatedData.method === "CRYPTO" && !validatedData.cryptoType) {
        return DashboardController.sendError(
          res,
          400,
          "Crypto type is required for crypto deposits",
          "VALIDATION_ERROR"
        );
      }
      const reference = `DP${Date.now()}${Math.random().toString(36).substr(2, 6).toUpperCase()}`;
      const depositData = {
        user_id: req.user.id,
        amount: validatedData.amount,
        method: validatedData.method,
        paypal_email: validatedData.paypalEmail,
        crypto_type: validatedData.cryptoType,
        notes: validatedData.note,
        reference,
        status: "PENDING"
      };
      const { data: deposit, error } = await supabase.from("deposits").insert(depositData).select().single();
      if (error) throw error;
      res.status(201).json({
        success: true,
        message: "Deposit request created successfully",
        data: {
          deposit: {
            ...deposit,
            amount: Number(deposit.amount)
          }
        }
      });
    } catch (error) {
      console.error("Create deposit error:", error);
      if (error instanceof z$1.ZodError) {
        return DashboardController.sendValidationError(res, error);
      }
      DashboardController.sendError(
        res,
        500,
        "Failed to create deposit",
        "CREATE_DEPOSIT_ERROR"
      );
    }
  }
  /**
   * Get user deposits
   */
  static async getDeposits(req, res) {
    try {
      if (!req.user) {
        return DashboardController.sendError(res, 401, "Authentication required", "AUTH_REQUIRED");
      }
      const { status, method, page = 1, limit = 20 } = req.query;
      const offset = (Number(page) - 1) * Number(limit);
      const query = supabase.from("deposits").select("*", { count: "exact" }).eq("user_id", req.user.id).order("created_at", { ascending: false }).range(offset, offset + Number(limit) - 1);
      if (status) query.eq("status", status);
      if (method) query.eq("method", method);
      const { data: deposits, error, count } = await query;
      if (error) throw error;
      res.json({
        success: true,
        data: {
          deposits: deposits?.map((d) => ({
            ...d,
            amount: Number(d.amount)
          })) || [],
          pagination: {
            page: Number(page),
            limit: Number(limit),
            total: count || 0,
            pages: Math.ceil((count || 0) / Number(limit))
          }
        }
      });
    } catch (error) {
      console.error("Get deposits error:", error);
      DashboardController.sendError(res, 500, "Failed to get deposits", "DEPOSITS_ERROR");
    }
  }
  /**
   * Get single deposit by ID
   */
  static async getDepositById(req, res) {
    try {
      if (!req.user) {
        return DashboardController.sendError(res, 401, "Authentication required", "AUTH_REQUIRED");
      }
      const { id } = req.params;
      const { data: deposit, error } = await supabase.from("deposits").select("*").eq("id", id).eq("user_id", req.user.id).single();
      if (error || !deposit) {
        return DashboardController.sendError(res, 404, "Deposit not found", "DEPOSIT_NOT_FOUND");
      }
      res.json({
        success: true,
        data: {
          deposit: {
            ...deposit,
            amount: Number(deposit.amount)
          }
        }
      });
    } catch (error) {
      console.error("Get deposit error:", error);
      DashboardController.sendError(res, 500, "Failed to get deposit", "DEPOSIT_ERROR");
    }
  }
  /**
   * Cancel pending deposit
   */
  static async cancelDeposit(req, res) {
    try {
      if (!req.user) {
        return DashboardController.sendError(res, 401, "Authentication required", "AUTH_REQUIRED");
      }
      const { id } = req.params;
      const { reason } = req.body;
      const { data: deposit, error: fetchError } = await supabase.from("deposits").select("*").eq("id", id).eq("user_id", req.user.id).single();
      if (fetchError || !deposit) {
        return DashboardController.sendError(res, 404, "Deposit not found", "DEPOSIT_NOT_FOUND");
      }
      if (!["PENDING", "PROCESSING"].includes(deposit.status)) {
        return DashboardController.sendError(
          res,
          400,
          "Deposit cannot be cancelled in current status",
          "INVALID_STATUS"
        );
      }
      const { data: updatedDeposit, error } = await supabase.from("deposits").update({
        status: "CANCELLED",
        cancelled_at: (/* @__PURE__ */ new Date()).toISOString(),
        cancellation_reason: reason
      }).eq("id", id).select().single();
      if (error) throw error;
      res.json({
        success: true,
        message: "Deposit cancelled successfully",
        data: { deposit: updatedDeposit }
      });
    } catch (error) {
      console.error("Cancel deposit error:", error);
      DashboardController.sendError(res, 500, "Failed to cancel deposit", "CANCEL_DEPOSIT_ERROR");
    }
  }
  /**
   * Admin: Update deposit status
   */
  static async updateDepositStatus(req, res) {
    try {
      const { id } = req.params;
      const validatedData = updateDepositStatusSchema.parse(req.body);
      const { data: deposit, error: fetchError } = await supabase.from("deposits").select("*").eq("id", id).single();
      if (fetchError || !deposit) {
        return DashboardController.sendError(res, 404, "Deposit not found", "DEPOSIT_NOT_FOUND");
      }
      const updateData = {
        status: validatedData.status,
        admin_notes: validatedData.adminNotes,
        transaction_hash: validatedData.transactionHash,
        failure_reason: validatedData.failureReason
      };
      if (validatedData.status === "COMPLETED") {
        updateData.completed_at = (/* @__PURE__ */ new Date()).toISOString();
      } else if (validatedData.status === "FAILED") {
        updateData.failed_at = (/* @__PURE__ */ new Date()).toISOString();
      }
      const { data: updatedDeposit, error } = await supabase.from("deposits").update(updateData).eq("id", id).select().single();
      if (error) throw error;
      res.json({
        success: true,
        message: "Deposit status updated successfully",
        data: { deposit: updatedDeposit }
      });
    } catch (error) {
      console.error("Update deposit status error:", error);
      if (error instanceof z$1.ZodError) {
        return DashboardController.sendValidationError(res, error);
      }
      DashboardController.sendError(
        res,
        500,
        "Failed to update deposit status",
        "UPDATE_DEPOSIT_ERROR"
      );
    }
  }
  /**
   * Create withdrawal request
   */
  static async createWithdrawal(req, res) {
    try {
      if (!req.user) {
        return DashboardController.sendError(res, 401, "Authentication required", "AUTH_REQUIRED");
      }
      const validatedData = createWithdrawalSchema.parse(req.body);
      const { data: user, error: userError } = await supabase.from("users").select("current_balance").eq("id", req.user.id).single();
      if (userError || !user || Number(user.current_balance) < validatedData.amount) {
        return DashboardController.sendError(res, 400, "Insufficient balance", "INSUFFICIENT_BALANCE");
      }
      const reference = `WD${Date.now()}${Math.random().toString(36).substr(2, 5).toUpperCase()}`;
      const { data: withdrawal, error } = await supabase.from("withdrawals").insert({
        id: req.user.id,
        amount: validatedData.amount,
        method: validatedData.method,
        bank_account: validatedData.bankAccount,
        crypto_address: validatedData.cryptoAddress,
        notes: validatedData.notes,
        reference,
        status: "PENDING"
      }).select().single();
      if (error) throw error;
      res.status(201).json({
        success: true,
        message: "Withdrawal request created successfully",
        data: { withdrawal }
      });
    } catch (error) {
      console.error("Create withdrawal error:", error);
      if (error instanceof z$1.ZodError) {
        return DashboardController.sendValidationError(res, error);
      }
      DashboardController.sendError(
        res,
        500,
        "Failed to create withdrawal",
        "CREATE_WITHDRAWAL_ERROR"
      );
    }
  }
  // HELPERS
  static async getUserPortfolios(userId, withHoldings = false) {
    try {
      const query = supabase.from("portfolios").select(
        withHoldings ? `*, holdings:holdings(id, symbol, name, quantity, avgPrice, current_price, total_value, pnl, pnl_percent, asset_type)` : `id, name, total_value, total_invested, total_returns, performance, is_default`
      ).eq("id", userId).order("is_default", { ascending: false }).order("created_at", { ascending: false });
      const { data: portfolios, error } = await query;
      if (error) throw error;
      return (portfolios || []).map((p) => ({
        ...p,
        totalValue: Number(p.total_value || 0),
        totalInvested: Number(p.total_invested || 0),
        totalReturns: Number(p.total_returns || 0),
        performance: Number(p.performance || 0),
        ...withHoldings && {
          holdings: p.holdings?.map((h) => ({
            ...h,
            quantity: Number(h.quantity || 0),
            avgPrice: Number(h.avg_price || 0),
            currentPrice: Number(h.current_price || 0),
            totalValue: Number(h.total_value || 0),
            pnl: Number(h.pnl || 0),
            pnlPercent: Number(h.pnl_percent || 0)
          })) || []
        }
      }));
    } catch (error) {
      console.error("Error fetching user portfolios:", error);
      return [];
    }
  }
  static async getRecentTransactions(userId) {
    try {
      const { data: transactions, error } = await supabase.from("transactions").select("id, type, symbol, amount, status, created_at, executed_at").eq("user_id", userId).order("created_at", { ascending: false }).limit(10);
      if (error) throw error;
      return (transactions || []).map((t) => ({
        ...t,
        amount: Number(t.amount || 0)
      }));
    } catch (error) {
      console.error("Error fetching recent transactions:", error);
      return [];
    }
  }
  static async getActiveInvestments(userId) {
    try {
      const { data: investments, error } = await supabase.from("investments").select(
        `id, name, amount, current_value, returns, return_percent, start_date, end_date, risk_level`
      ).eq("user_id", userId).eq("status", "ACTIVE");
      if (error) throw error;
      return (investments || []).map((i) => ({
        ...i,
        amount: Number(i.amount || 0),
        currentValue: Number(i.current_value || 0),
        returns: Number(i.returns || 0),
        returnPercent: Number(i.return_percent || 0)
      }));
    } catch (error) {
      console.error("Error fetching active investments:", error);
      return [];
    }
  }
  static async getPendingWithdrawals(userId) {
    try {
      const { data: withdrawals, error } = await supabase.from("withdrawals").select("id, amount, method, created_at").eq("user_id", userId).eq("status", "PENDING");
      if (error) throw error;
      return (withdrawals || []).map((w) => ({
        ...w,
        amount: Number(w.amount || 0)
      }));
    } catch (error) {
      console.error("Error fetching pending withdrawals:", error);
      return [];
    }
  }
  static async getMonthlyPerformance(userId) {
    try {
      const monthlyData = [];
      for (let i = 11; i >= 0; i--) {
        const date = /* @__PURE__ */ new Date();
        date.setMonth(date.getMonth() - i);
        const performance = Math.random() * 20 - 5;
        monthlyData.push({
          month: date.toISOString().substring(0, 7),
          // YYYY-MM format
          performance: Number(performance.toFixed(2))
        });
      }
      return monthlyData;
    } catch (error) {
      console.error("Error fetching monthly performance:", error);
      return [];
    }
  }
  static async getRiskDistribution(userId) {
    try {
      const { data: distribution, error } = await supabase.rpc("get_risk_distribution", { user_id: userId });
      if (error) {
        console.warn("get_risk_distribution RPC not found, returning empty array");
        return [];
      }
      return distribution || [];
    } catch (error) {
      console.error("Error fetching risk distribution:", error);
      return [];
    }
  }
  static sendError(res, status, message, code) {
    res.status(status).json({
      success: false,
      message,
      code
    });
  }
  static sendValidationError(res, error) {
    res.status(400).json({
      success: false,
      message: "Validation error",
      errors: error.errors.map((err) => ({
        field: err.path.join("."),
        message: err.message
      }))
    });
  }
}
const {
  getDashboardOverview,
  getPortfolios,
  getInvestments,
  getTransactions,
  getPackages,
  createInvestment,
  createWithdrawal,
  getDepositById,
  getDepositDetails,
  getDeposits,
  createDeposit,
  cancelDeposit,
  updateDepositStatus
} = DashboardController;
const router$1 = express__default.Router();
router$1.use(authenticate);
router$1.get("/overview", getDashboardOverview);
router$1.get("/portfolios", getPortfolios);
router$1.get("/packages", getPackages);
router$1.get("/investments", getInvestments);
router$1.get("/transactions", getTransactions);
router$1.post("/investments", createInvestment);
router$1.post("/withdrawal", createWithdrawal);
router$1.get("/deposit-details", getDepositDetails);
router$1.post("/deposits", createDeposit);
router$1.get("/deposits", getDeposits);
router$1.get("/deposits/:id", getDepositById);
router$1.put("/deposits/:id/cancel", cancelDeposit);
router$1.put("/deposits/:id/status", updateDepositStatus);
const ADMIN_ROLE = "ADMIN";
const SENSITIVE_USER_FIELDS = /* @__PURE__ */ new Set(["password", "two_factor_secret"]);
function apiSuccess(res, data, status = 200) {
  return res.status(status).json({ success: true, data });
}
function apiError(res, message, code, status = 400, meta) {
  const body = { success: false, message, code };
  if (meta) body.meta = meta;
  return res.status(status).json(body);
}
function sanitizeUser(user) {
  if (!user) return null;
  const copy = { ...user };
  for (const f of SENSITIVE_USER_FIELDS) {
    if (f in copy) delete copy[f];
  }
  return copy;
}
async function writeAdminLog(adminId, action, targetUserId, details, ip) {
  try {
    await supabase.from("admin_logs").insert({
      admin_id: adminId ?? null,
      action,
      target_user_id: targetUserId,
      details: JSON.stringify(details ?? {}),
      ip_address: ip ?? null,
      created_at: (/* @__PURE__ */ new Date()).toISOString()
    });
  } catch (err) {
    console.error("admin log error", err);
  }
}
const PaginationSchema = z$1.object({
  page: z$1.string().optional().transform((val) => val ? Number(val) : 1).refine((n) => Number.isInteger(n) && n >= 1, { message: "page must be integer >=1" }),
  perPage: z$1.string().optional().transform((val) => val ? Number(val) : 50).refine((n) => Number.isInteger(n) && n >= 1 && n <= 500, { message: "perPage must be 1..500" })
});
function pageToRange(page = 1, perPage = 50) {
  const p = Math.max(1, Math.trunc(Number(page) || 1));
  const pp = Math.min(500, Math.max(1, Math.trunc(Number(perPage) || 50)));
  const from = (p - 1) * pp;
  const to = from + pp - 1;
  return { from, to, page: p, perPage: pp };
}
const GetAllUsersQuerySchema = z$1.object({
  page: PaginationSchema.shape.page.optional(),
  perPage: PaginationSchema.shape.perPage.optional(),
  role: z$1.string().optional(),
  status: z$1.string().optional(),
  q: z$1.string().optional()
});
z$1.object({
  email: z$1.string().email().optional(),
  username: z$1.string().min(1).optional(),
  first_name: z$1.string().optional().nullable(),
  last_name: z$1.string().optional().nullable(),
  phone: z$1.string().optional().nullable(),
  avatar: z$1.string().url().optional().nullable(),
  role: z$1.string().optional(),
  status: z$1.string().optional(),
  email_verified: z$1.boolean().optional(),
  two_factor_enabled: z$1.boolean().optional(),
  total_investment: z$1.number().optional(),
  current_balance: z$1.number().optional(),
  total_returns: z$1.number().optional(),
  risk_tolerance: z$1.string().optional(),
  last_login_at: z$1.string().optional().nullable(),
  id_verified: z$1.boolean().optional(),
  id_verification_status: z$1.string().optional()
}).strict();
z$1.object({
  email: z$1.string().email(),
  username: z$1.string().min(1).optional(),
  password: z$1.string().min(8),
  // admin-created password; hashed
  first_name: z$1.string().optional().nullable(),
  last_name: z$1.string().optional().nullable(),
  phone: z$1.string().optional().nullable(),
  avatar: z$1.string().url().optional().nullable(),
  role: z$1.string().optional()
});
z$1.object({
  userId: z$1.string().uuid(),
  amount: z$1.number(),
  // positive or negative
  reason: z$1.string().min(1),
  createTransaction: z$1.boolean().optional().default(true)
});
z$1.object({
  userId: z$1.string().uuid(),
  comment: z$1.string().optional().nullable()
});
z$1.object({
  page: PaginationSchema.shape.page.optional(),
  perPage: PaginationSchema.shape.perPage.optional(),
  adminId: z$1.string().optional(),
  action: z$1.string().optional()
});
const getAllUsers = async (req, res) => {
  try {
    console.log("📥 getAllUsers called");
    if (req.user?.role !== ADMIN_ROLE) {
      return apiError(res, "Admin required", "ADMIN_REQUIRED", 403);
    }
    const parsed = GetAllUsersQuerySchema.safeParse(req.query ?? {});
    if (!parsed.success) {
      return apiError(res, "Invalid query", "INVALID_QUERY", 400, parsed.error.errors);
    }
    const { page = 1, perPage = 20, role, status, q } = parsed.data;
    const { from, to } = pageToRange(page, perPage);
    let query = supabase.from("users").select("*", { count: "exact" }).order("created_at", { ascending: false });
    query = query.or("role.neq.ADMIN,role.is.null");
    if (role && role.toUpperCase() !== "ADMIN") {
      query = query.eq("role", role.toUpperCase());
    }
    if (status) {
      query = query.eq("status", status.toUpperCase());
    }
    if (q && String(q).trim().length >= 2) {
      const like = `%${String(q).trim().toLowerCase()}%`;
      query = query.or(
        `email.ilike.${like},username.ilike.${like},first_name.ilike.${like},last_name.ilike.${like}`
      );
    }
    query = query.range(from, to);
    const { data, count, error } = await query;
    if (error) {
      console.error("❌ Supabase error:", error);
      return apiError(res, "Database error", "DB_ERROR", 500);
    }
    const sanitized = (data || []).map(sanitizeUser);
    const totalCount = typeof count === "number" ? count : sanitized.length;
    return res.status(200).json({
      success: true,
      message: "Users fetched successfully",
      data: sanitized,
      meta: {
        page,
        perPage,
        total: totalCount,
        totalPages: Math.ceil(totalCount / perPage)
      }
    });
  } catch (err) {
    console.error("❌ Unexpected error in getAllUsers:", err);
    return apiError(res, "Internal error", "INTERNAL_ERROR", 500);
  }
};
async function getUserDetails(req, res) {
  try {
    if (req.user?.role !== ADMIN_ROLE) {
      return apiError(res, "Admin required", "ADMIN_REQUIRED", 403);
    }
    const userId = req.params.id;
    if (!userId) {
      return apiError(res, "User id required", "USER_ID_REQUIRED", 400);
    }
    const { data: user, error: userErr } = await supabase.from("users").select("*").eq("id", userId).maybeSingle();
    if (userErr) {
      console.error("Error fetching user:", userErr);
      return apiError(res, "Database error", "DB_ERROR", 500);
    }
    if (!user) {
      return apiError(res, "User not found", "USER_NOT_FOUND", 404);
    }
    async function fetchRelated(tableName, page = 1, perPage = 100) {
      const from = (page - 1) * perPage;
      const to = from + perPage - 1;
      let query = supabase.from(tableName).select("*", { count: "exact" }).eq("user_id", userId).order("created_at", { ascending: false }).range(from, to);
      if (tableName === "user_packages") {
        query = supabase.from(tableName).select(`
            *,
            packages:package_id (*)
          `, { count: "exact" }).eq("user_id", userId).order("subscribed_at", { ascending: false }).range(from, to);
      }
      const { data, count, error } = await query;
      if (error) {
        console.error(`Error fetching ${tableName}:`, error);
        return { data: [], count: 0, page, perPage };
      }
      return { data: data || [], count: count || 0, page, perPage };
    }
    const [transactions, deposits, withdrawals, investments, userPackages] = await Promise.all([
      fetchRelated("transactions"),
      fetchRelated("deposits"),
      fetchRelated("withdrawals"),
      fetchRelated("investments"),
      fetchRelated("user_packages")
    ]);
    const response = {
      user: sanitizeUser(user),
      related: {
        transactions: { items: transactions.data, meta: { total: transactions.count, page: transactions.page, perPage: transactions.perPage } },
        deposits: { items: deposits.data, meta: { total: deposits.count, page: deposits.page, perPage: deposits.perPage } },
        withdrawals: { items: withdrawals.data, meta: { total: withdrawals.count, page: withdrawals.page, perPage: withdrawals.perPage } },
        investments: { items: investments.data, meta: { total: investments.count, page: investments.page, perPage: investments.perPage } },
        user_packages: { items: userPackages.data, meta: { total: userPackages.count, page: userPackages.page, perPage: userPackages.perPage } }
      }
    };
    return apiSuccess(res, response);
  } catch (err) {
    console.error("getUserDetails error:", err);
    return apiError(res, "Internal error", "INTERNAL_ERROR", 500);
  }
}
async function updateUser(req, res) {
  try {
    if (req.user?.role !== ADMIN_ROLE) {
      return apiError(res, "Admin required", "ADMIN_REQUIRED", 403);
    }
    const userId = req.params.id;
    if (!userId) {
      return apiError(res, "User id required", "USER_ID_REQUIRED", 400);
    }
    const forbiddenFields = ["password", "two_factor_secret", "id_number", "id_front_image_url", "id_back_image_url", "accountBalance", "totalDeposit", "totalProfit", "totalWithdrawals", "kycStatus", "memberSince", "country", "lastLogin", "name"];
    const cleanBody = Object.fromEntries(
      Object.entries(req.body || {}).filter(([key]) => !forbiddenFields.includes(key))
    );
    const updates = cleanBody;
    if (!updates || Object.keys(updates).length === 0) {
      return apiError(res, "No updates provided", "NO_UPDATES", 400);
    }
    updates.updated_at = (/* @__PURE__ */ new Date()).toISOString();
    const { data: updatedUser, error: updateErr } = await supabase.from("users").update(updates).eq("id", userId).select("*").maybeSingle();
    if (updateErr) {
      console.error("Update error:", updateErr);
      return apiError(res, "Database error", "DB_ERROR", 500);
    }
    if (!updatedUser) {
      return apiError(res, "User not found", "USER_NOT_FOUND", 404);
    }
    await writeAdminLog(req.user?.id, "UPDATE_USER", userId, { updates }, req.ip);
    return apiSuccess(res, sanitizeUser(updatedUser));
  } catch (err) {
    console.error("updateUser error:", err);
    return apiError(res, "Internal error", "INTERNAL_ERROR", 500);
  }
}
async function createUser(req, res) {
  try {
    if (req.user?.role !== ADMIN_ROLE) {
      return apiError(res, "Admin required", "ADMIN_REQUIRED", 403);
    }
    const CreateUserSchema = z$1.object({
      email: z$1.string().email(),
      username: z$1.string().min(1).optional(),
      password: z$1.string().min(8),
      first_name: z$1.string().optional().nullable(),
      last_name: z$1.string().optional().nullable(),
      phone: z$1.string().optional().nullable(),
      avatar: z$1.string().url().optional().nullable(),
      role: z$1.string().optional()
    });
    const parsed = CreateUserSchema.safeParse(req.body);
    if (!parsed.success) {
      return apiError(res, "Invalid payload", "VALIDATION_ERROR", 400, parsed.error.errors);
    }
    const { email, password, username, first_name, last_name, phone, avatar, role } = parsed.data;
    const { data: emailExists, error: emailErr } = await supabase.from("users").select("id").eq("email", email).maybeSingle();
    if (emailErr) {
      return apiError(res, "Database error", "DB_ERROR", 500);
    }
    if (emailExists) {
      return apiError(res, "Email already in use", "EMAIL_TAKEN", 409);
    }
    if (username) {
      const { data: userExists, error: userErr } = await supabase.from("users").select("id").eq("username", username).maybeSingle();
      if (userErr) {
        return apiError(res, "Database error", "DB_ERROR", 500);
      }
      if (userExists) {
        return apiError(res, "Username taken", "USERNAME_TAKEN", 409);
      }
    }
    const hashed = await bcrypt.hash(password, 12);
    const { data: createdUser, error: insertErr } = await supabase.from("users").insert({
      email,
      password: hashed,
      username: username ?? null,
      first_name: first_name ?? null,
      last_name: last_name ?? null,
      phone: phone ?? null,
      avatar: avatar ?? null,
      role: role ?? "USER",
      created_at: (/* @__PURE__ */ new Date()).toISOString(),
      updated_at: (/* @__PURE__ */ new Date()).toISOString()
    }).select("*").maybeSingle();
    if (insertErr) {
      console.error("Insert error:", insertErr);
      return apiError(res, "Database error", "DB_ERROR", 500);
    }
    await writeAdminLog(req.user?.id, "CREATE_USER", createdUser.id, { email, username }, req.ip);
    return apiSuccess(res, sanitizeUser(createdUser), 201);
  } catch (err) {
    console.error("createUser error:", err);
    return apiError(res, "Internal error", "INTERNAL_ERROR", 500);
  }
}
async function deleteUser(req, res) {
  try {
    if (req.user?.role !== ADMIN_ROLE) {
      return apiError(res, "Admin required", "ADMIN_REQUIRED", 403);
    }
    const userId = req.params.id;
    if (!userId) {
      return apiError(res, "User id required", "USER_ID_REQUIRED", 400);
    }
    const { data: existing, error: existErr } = await supabase.from("users").select("id,status").eq("id", userId).maybeSingle();
    if (existErr) {
      return apiError(res, "Database error", "DB_ERROR", 500);
    }
    if (!existing) {
      return apiError(res, "User not found", "USER_NOT_FOUND", 404);
    }
    const { data: updated, error } = await supabase.from("users").update({ status: "INACTIVE", updated_at: (/* @__PURE__ */ new Date()).toISOString() }).eq("id", userId).select("*").maybeSingle();
    if (error) {
      return apiError(res, "Database error", "DB_ERROR", 500);
    }
    await writeAdminLog(req.user?.id, "DELETE_USER", userId, { previousStatus: existing.status }, req.ip);
    return apiSuccess(res, sanitizeUser(updated));
  } catch (err) {
    console.error("deleteUser error:", err);
    return apiError(res, "Internal error", "INTERNAL_ERROR", 500);
  }
}
async function suspendUser(req, res) {
  try {
    if (req.user?.role !== ADMIN_ROLE) {
      return apiError(res, "Admin required", "ADMIN_REQUIRED", 403);
    }
    const userId = req.params.id;
    if (!userId) {
      return apiError(res, "User id required", "USER_ID_REQUIRED", 400);
    }
    const { data: updated, error } = await supabase.from("users").update({ status: "SUSPENDED", updated_at: (/* @__PURE__ */ new Date()).toISOString() }).eq("id", userId).select("*").maybeSingle();
    if (error) {
      return apiError(res, "Database error", "DB_ERROR", 500);
    }
    if (!updated) {
      return apiError(res, "User not found", "USER_NOT_FOUND", 404);
    }
    await writeAdminLog(req.user?.id, "SUSPEND_USER", userId, {}, req.ip);
    return apiSuccess(res, sanitizeUser(updated));
  } catch (err) {
    console.error("suspendUser error:", err);
    return apiError(res, "Internal error", "INTERNAL_ERROR", 500);
  }
}
async function restoreUser(req, res) {
  try {
    if (req.user?.role !== ADMIN_ROLE) {
      return apiError(res, "Admin required", "ADMIN_REQUIRED", 403);
    }
    const userId = req.params.id;
    if (!userId) {
      return apiError(res, "User id required", "USER_ID_REQUIRED", 400);
    }
    const { data: updated, error } = await supabase.from("users").update({ status: "ACTIVE", updated_at: (/* @__PURE__ */ new Date()).toISOString() }).eq("id", userId).select("*").maybeSingle();
    if (error) {
      return apiError(res, "Database error", "DB_ERROR", 500);
    }
    if (!updated) {
      return apiError(res, "User not found", "USER_NOT_FOUND", 404);
    }
    await writeAdminLog(req.user?.id, "RESTORE_USER", userId, {}, req.ip);
    return apiSuccess(res, sanitizeUser(updated));
  } catch (err) {
    console.error("restoreUser error:", err);
    return apiError(res, "Internal error", "INTERNAL_ERROR", 500);
  }
}
async function changeUserRole(req, res) {
  try {
    if (req.user?.role !== ADMIN_ROLE) {
      return apiError(res, "Admin required", "ADMIN_REQUIRED", 403);
    }
    const userId = req.params.id;
    if (!userId) {
      return apiError(res, "User id required", "USER_ID_REQUIRED", 400);
    }
    const { role } = z$1.object({ role: z$1.string().min(1) }).parse(req.body);
    const { data: updated, error } = await supabase.from("users").update({ role, updated_at: (/* @__PURE__ */ new Date()).toISOString() }).eq("id", userId).select("*").maybeSingle();
    if (error) {
      return apiError(res, "Database error", "DB_ERROR", 500);
    }
    if (!updated) {
      return apiError(res, "User not found", "USER_NOT_FOUND", 404);
    }
    await writeAdminLog(req.user?.id, "CHANGE_ROLE", userId, { role }, req.ip);
    return apiSuccess(res, sanitizeUser(updated));
  } catch (err) {
    console.error("changeUserRole error:", err);
    return apiError(res, "Internal error", "INTERNAL_ERROR", 500);
  }
}
async function approveKyc(req, res) {
  try {
    if (req.user?.role !== ADMIN_ROLE) {
      return apiError(res, "Admin required", "ADMIN_REQUIRED", 403);
    }
    const { userId, comment } = z$1.object({
      userId: z$1.string().uuid(),
      comment: z$1.string().optional().nullable()
    }).parse(req.body);
    const { data: updated, error } = await supabase.from("users").update({
      id_verified: true,
      id_verification_status: "verified",
      id_reviewed_at: (/* @__PURE__ */ new Date()).toISOString(),
      updated_at: (/* @__PURE__ */ new Date()).toISOString()
    }).eq("id", userId).select("*").maybeSingle();
    if (error) {
      return apiError(res, "Database error", "DB_ERROR", 500);
    }
    if (!updated) {
      return apiError(res, "User not found", "USER_NOT_FOUND", 404);
    }
    await writeAdminLog(req.user?.id, "APPROVE_KYC", userId, { comment }, req.ip);
    return apiSuccess(res, sanitizeUser(updated));
  } catch (err) {
    console.error("approveKyc error:", err);
    return apiError(res, "Internal error", "INTERNAL_ERROR", 500);
  }
}
async function rejectKyc(req, res) {
  try {
    if (req.user?.role !== ADMIN_ROLE) {
      return apiError(res, "Admin required", "ADMIN_REQUIRED", 403);
    }
    const { userId, comment } = z$1.object({
      userId: z$1.string().uuid(),
      comment: z$1.string().optional().nullable()
    }).parse(req.body);
    const { data: updated, error } = await supabase.from("users").update({
      id_verified: false,
      id_verification_status: "rejected",
      id_reviewed_at: (/* @__PURE__ */ new Date()).toISOString(),
      updated_at: (/* @__PURE__ */ new Date()).toISOString()
    }).eq("id", userId).select("*").maybeSingle();
    if (error) {
      return apiError(res, "Database error", "DB_ERROR", 500);
    }
    if (!updated) {
      return apiError(res, "User not found", "USER_NOT_FOUND", 404);
    }
    await writeAdminLog(req.user?.id, "REJECT_KYC", userId, { comment }, req.ip);
    return apiSuccess(res, sanitizeUser(updated));
  } catch (err) {
    console.error("rejectKyc error:", err);
    return apiError(res, "Internal error", "INTERNAL_ERROR", 500);
  }
}
async function fetchRelatedGeneric(req, res, table) {
  try {
    if (req.user?.role !== ADMIN_ROLE) {
      return apiError(res, "Admin required", "ADMIN_REQUIRED", 403);
    }
    const userId = req.params.id;
    if (!userId) {
      return apiError(res, "User id required", "USER_ID_REQUIRED", 400);
    }
    const { page = 1, perPage = 50 } = z$1.object({
      page: z$1.string().optional().transform((v) => Number(v) || 1),
      perPage: z$1.string().optional().transform((v) => Number(v) || 50)
    }).parse(req.query);
    const { from, to } = pageToRange(page, perPage);
    const { data, count, error } = await supabase.from(table).select("*", { count: "exact" }).eq("user_id", userId).order("created_at", { ascending: false }).range(from, to);
    if (error) {
      return apiError(res, "Database error", "DB_ERROR", 500);
    }
    const items = data || [];
    const totalCount = typeof count === "number" ? count : items.length;
    return res.status(200).json({
      success: true,
      data: items,
      meta: { page, perPage, total: totalCount, totalPages: Math.ceil(totalCount / perPage) }
    });
  } catch (err) {
    console.error(`fetch${table} error:`, err);
    return apiError(res, "Internal error", "INTERNAL_ERROR", 500);
  }
}
async function getUserTransactions(req, res) {
  return fetchRelatedGeneric(req, res, "transactions");
}
async function getUserDeposits(req, res) {
  return fetchRelatedGeneric(req, res, "deposits");
}
async function getUserWithdrawals(req, res) {
  return fetchRelatedGeneric(req, res, "withdrawals");
}
async function getUserInvestments(req, res) {
  return fetchRelatedGeneric(req, res, "investments");
}
async function getUserPackages(req, res) {
  try {
    if (req.user?.role !== ADMIN_ROLE) {
      return apiError(res, "Admin required", "ADMIN_REQUIRED", 403);
    }
    const userId = req.params.id;
    if (!userId) {
      return apiError(res, "User id required", "USER_ID_REQUIRED", 400);
    }
    const { page = 1, perPage = 50 } = z$1.object({
      page: z$1.string().optional().transform((v) => Number(v) || 1),
      perPage: z$1.string().optional().transform((v) => Number(v) || 50)
    }).parse(req.query);
    const { from, to } = pageToRange(page, perPage);
    const { data, count, error } = await supabase.from("user_packages").select(`
        *,
        packages:package_id (*)
      `, { count: "exact" }).eq("user_id", userId).order("subscribed_at", { ascending: false }).range(from, to);
    if (error) {
      return apiError(res, "Database error", "DB_ERROR", 500);
    }
    const items = data || [];
    const totalCount = typeof count === "number" ? count : items.length;
    return res.status(200).json({
      success: true,
      data: items,
      meta: { page, perPage, total: totalCount, totalPages: Math.ceil(totalCount / perPage) }
    });
  } catch (err) {
    console.error("getUserPackages error:", err);
    return apiError(res, "Internal error", "INTERNAL_ERROR", 500);
  }
}
async function adjustBalance(req, res) {
  try {
    if (req.user?.role !== ADMIN_ROLE) {
      return apiError(res, "Admin required", "ADMIN_REQUIRED", 403);
    }
    const { userId, amount, reason } = z$1.object({
      userId: z$1.string().uuid(),
      amount: z$1.number(),
      reason: z$1.string().min(1)
    }).parse(req.body);
    const { data: user, error: userErr } = await supabase.from("users").select("current_balance").eq("id", userId).maybeSingle();
    if (userErr || !user) {
      return apiError(res, "User not found", "USER_NOT_FOUND", 404);
    }
    const newBalance = (Number(user.current_balance) || 0) + Number(amount);
    const { data: updatedUser, error: updErr } = await supabase.from("users").update({ current_balance: newBalance, updated_at: (/* @__PURE__ */ new Date()).toISOString() }).eq("id", userId).select("*").maybeSingle();
    if (updErr) {
      return apiError(res, "Database error", "DB_ERROR", 500);
    }
    await writeAdminLog(req.user?.id, "ADJUST_BALANCE", userId, { amount, reason }, req.ip);
    return apiSuccess(res, { user: sanitizeUser(updatedUser) });
  } catch (err) {
    console.error("adjustBalance error:", err);
    return apiError(res, "Internal error", "INTERNAL_ERROR", 500);
  }
}
async function getAdminLogs(req, res) {
  try {
    if (req.user?.role !== ADMIN_ROLE) {
      return apiError(res, "Admin required", "ADMIN_REQUIRED", 403);
    }
    const { page = 1, perPage = 50, adminId, action } = z$1.object({
      page: z$1.string().optional().transform((v) => Number(v) || 1),
      perPage: z$1.string().optional().transform((v) => Number(v) || 50),
      adminId: z$1.string().optional(),
      action: z$1.string().optional()
    }).parse(req.query);
    const { from, to } = pageToRange(page, perPage);
    let query = supabase.from("admin_logs").select("*", { count: "exact" }).order("created_at", { ascending: false }).range(from, to);
    if (adminId) query = query.eq("admin_id", adminId);
    if (action) query = query.eq("action", action);
    const { data, count, error } = await query;
    if (error) {
      return apiError(res, "Database error", "DB_ERROR", 500);
    }
    const logs = data || [];
    const totalCount = typeof count === "number" ? count : logs.length;
    return res.status(200).json({
      success: true,
      data: logs,
      meta: { page, perPage, total: totalCount, totalPages: Math.ceil(totalCount / perPage) }
    });
  } catch (err) {
    console.error("getAdminLogs error:", err);
    return apiError(res, "Internal error", "INTERNAL_ERROR", 500);
  }
}
async function exportUsersCSV(req, res) {
  try {
    if (req.user?.role !== ADMIN_ROLE) {
      return apiError(res, "Admin required", "ADMIN_REQUIRED", 403);
    }
    const { role, status, q } = GetAllUsersQuerySchema.parse(req.query);
    let query = supabase.from("users").select("id,email,username,first_name,last_name,phone,role,status,created_at,updated_at");
    if (role) query = query.eq("role", role);
    if (status) query = query.eq("status", status);
    if (q && String(q).trim().length >= 2) {
      const like = `%${String(q).trim().toLowerCase()}%`;
      query = query.or(`email.ilike.${like},username.ilike.${like},first_name.ilike.${like},last_name.ilike.${like}`);
    }
    const { data, error } = await query.order("created_at", { ascending: false });
    if (error) {
      return apiError(res, "Database error", "DB_ERROR", 500);
    }
    const rows = data || [];
    const headers = ["id", "email", "username", "first_name", "last_name", "phone", "role", "status", "created_at", "updated_at"];
    const csv = [headers.join(",")].concat(
      rows.map(
        (r) => headers.map((h) => {
          const v = r[h] === null || r[h] === void 0 ? "" : String(r[h]).replace(/"/g, '""');
          return v.includes(",") || v.includes('"') ? `"${v}"` : v;
        }).join(",")
      )
    ).join("\n");
    res.setHeader("Content-Type", "text/csv");
    res.setHeader("Content-Disposition", `attachment; filename="users_export_${(/* @__PURE__ */ new Date()).toISOString()}.csv"`);
    await writeAdminLog(req.user?.id, "EXPORT_USERS_CSV", null, { filters: req.query }, req.ip);
    return res.status(200).send(csv);
  } catch (err) {
    console.error("exportUsersCSV error:", err);
    return apiError(res, "Internal error", "INTERNAL_ERROR", 500);
  }
}
async function batchUpdateInvestments(req, res) {
  try {
    console.log("📥 batchUpdateInvestments called for user:", req.params.id);
    if (req.user?.role !== ADMIN_ROLE) {
      return apiError(res, "Admin required", "ADMIN_REQUIRED", 403);
    }
    const userId = req.params.id;
    if (!userId) {
      return apiError(res, "User id required", "USER_ID_REQUIRED", 400);
    }
    const BatchInvestmentsSchema = z$1.object({
      investments: z$1.array(z$1.object({
        id: z$1.string().optional(),
        package_id: z$1.string().optional().nullable(),
        name: z$1.string().min(1),
        description: z$1.string().optional().nullable(),
        amount: z$1.number().positive(),
        current_value: z$1.number(),
        returns: z$1.number().optional().default(0),
        return_percent: z$1.number().optional().default(0),
        duration: z$1.number().int().positive(),
        start_date: z$1.string(),
        end_date: z$1.string(),
        status: z$1.enum(["ACTIVE", "COMPLETED", "CANCELLED", "PAUSED"]).optional().default("ACTIVE"),
        risk_level: z$1.enum(["LOW", "MODERATE", "HIGH"]).optional().default("MODERATE")
      }))
    });
    const parsed = BatchInvestmentsSchema.safeParse(req.body ?? {});
    if (!parsed.success) {
      return apiError(res, "Invalid payload", "VALIDATION_ERROR", 400, parsed.error.errors);
    }
    const { investments } = parsed.data;
    const updates = investments.filter((inv) => inv.id && !inv.id.startsWith("inv_"));
    const inserts = investments.filter((inv) => !inv.id || inv.id.startsWith("inv_"));
    console.log(`🔄 Processing ${updates.length} updates and ${inserts.length} inserts`);
    const updatePromises = updates.map(async (inv) => {
      const updateData = {
        name: inv.name,
        description: inv.description,
        package_id: inv.package_id,
        amount: inv.amount,
        current_value: inv.current_value,
        returns: inv.returns,
        return_percent: inv.return_percent,
        duration: inv.duration,
        start_date: inv.start_date,
        end_date: inv.end_date,
        status: inv.status,
        risk_level: inv.risk_level,
        updated_at: (/* @__PURE__ */ new Date()).toISOString()
      };
      return supabase.from("investments").update(updateData).eq("id", inv.id).eq("user_id", userId).select().single();
    });
    const insertPromises = inserts.map(async (inv) => {
      const insertData = {
        user_id: userId,
        package_id: inv.package_id || null,
        name: inv.name,
        description: inv.description || null,
        amount: inv.amount,
        current_value: inv.current_value,
        returns: inv.returns,
        return_percent: inv.return_percent,
        duration: inv.duration,
        start_date: inv.start_date,
        end_date: inv.end_date,
        status: inv.status,
        risk_level: inv.risk_level,
        created_at: (/* @__PURE__ */ new Date()).toISOString(),
        updated_at: (/* @__PURE__ */ new Date()).toISOString()
      };
      return supabase.from("investments").insert(insertData).select().single();
    });
    const results = await Promise.allSettled([...updatePromises, ...insertPromises]);
    const successful = results.filter((r) => r.status === "fulfilled").length;
    const failed = results.filter((r) => r.status === "rejected").length;
    results.forEach((result, index) => {
      if (result.status === "rejected") {
        console.error(`Investment operation ${index} failed:`, result.reason);
      }
    });
    console.log(`✅ Batch update completed: ${successful} successful, ${failed} failed`);
    const { data: allInvestments, error } = await supabase.from("investments").select("*").eq("user_id", userId).order("created_at", { ascending: false });
    if (error) {
      console.error("Error fetching investments:", error);
      return apiError(res, "Failed to fetch updated investments", "DB_ERROR", 500);
    }
    await writeAdminLog(req.user?.id, "BATCH_UPDATE_INVESTMENTS", userId, {
      updates: updates.length,
      inserts: inserts.length,
      successful,
      failed
    }, req.ip);
    return apiSuccess(res, { investments: allInvestments || [], meta: { successful, failed } });
  } catch (err) {
    console.error("❌ batchUpdateInvestments error:", err);
    return apiError(res, err.message || "Internal error", "INTERNAL_ERROR", 500);
  }
}
async function batchUpdateTransactions(req, res) {
  try {
    console.log("📥 batchUpdateTransactions called for user:", req.params.id);
    if (req.user?.role !== ADMIN_ROLE) {
      return apiError(res, "Admin required", "ADMIN_REQUIRED", 403);
    }
    const userId = req.params.id;
    if (!userId) {
      return apiError(res, "User id required", "USER_ID_REQUIRED", 400);
    }
    const BatchTransactionsSchema = z$1.object({
      transactions: z$1.array(z$1.object({
        id: z$1.string().optional(),
        type: z$1.enum(["DEPOSIT", "WITHDRAWAL", "TRADE", "TRANSFER", "FEE", "REFUND"]),
        symbol: z$1.string().optional().nullable(),
        quantity: z$1.number().optional().nullable(),
        price: z$1.number().optional().nullable(),
        amount: z$1.number(),
        fee: z$1.number().optional().default(0),
        status: z$1.enum(["PENDING", "COMPLETED", "FAILED", "CANCELLED"]).optional().default("PENDING"),
        description: z$1.string().optional().nullable(),
        reference: z$1.string().optional().nullable(),
        executed_at: z$1.string().optional().nullable()
      }))
    });
    const parsed = BatchTransactionsSchema.safeParse(req.body ?? {});
    if (!parsed.success) {
      return apiError(res, "Invalid payload", "VALIDATION_ERROR", 400, parsed.error.errors);
    }
    const { transactions } = parsed.data;
    const updates = transactions.filter((tx) => tx.id && !tx.id.startsWith("tx_"));
    const inserts = transactions.filter((tx) => !tx.id || tx.id.startsWith("tx_"));
    console.log(`🔄 Processing ${updates.length} updates and ${inserts.length} inserts`);
    const updatePromises = updates.map(async (tx) => {
      const updateData = {
        type: tx.type,
        symbol: tx.symbol,
        quantity: tx.quantity,
        price: tx.price,
        amount: tx.amount,
        fee: tx.fee,
        status: tx.status,
        description: tx.description,
        reference: tx.reference,
        executed_at: tx.executed_at,
        updated_at: (/* @__PURE__ */ new Date()).toISOString()
      };
      return supabase.from("transactions").update(updateData).eq("id", tx.id).eq("user_id", userId).select().single();
    });
    const insertPromises = inserts.map(async (tx) => {
      const insertData = {
        user_id: userId,
        type: tx.type,
        symbol: tx.symbol || null,
        quantity: tx.quantity || null,
        price: tx.price || null,
        amount: tx.amount,
        fee: tx.fee,
        status: tx.status,
        description: tx.description || null,
        reference: tx.reference || null,
        executed_at: tx.executed_at || null,
        created_at: (/* @__PURE__ */ new Date()).toISOString(),
        updated_at: (/* @__PURE__ */ new Date()).toISOString()
      };
      return supabase.from("transactions").insert(insertData).select().single();
    });
    const results = await Promise.allSettled([...updatePromises, ...insertPromises]);
    const successful = results.filter((r) => r.status === "fulfilled").length;
    const failed = results.filter((r) => r.status === "rejected").length;
    results.forEach((result, index) => {
      if (result.status === "rejected") {
        console.error(`Transaction operation ${index} failed:`, result.reason);
      }
    });
    console.log(`✅ Batch update completed: ${successful} successful, ${failed} failed`);
    const { data: allTransactions, error } = await supabase.from("transactions").select("*").eq("user_id", userId).order("created_at", { ascending: false });
    if (error) {
      console.error("Error fetching transactions:", error);
      return apiError(res, "Failed to fetch updated transactions", "DB_ERROR", 500);
    }
    await writeAdminLog(req.user?.id, "BATCH_UPDATE_TRANSACTIONS", userId, {
      updates: updates.length,
      inserts: inserts.length,
      successful,
      failed
    }, req.ip);
    return apiSuccess(res, { transactions: allTransactions || [], meta: { successful, failed } });
  } catch (err) {
    console.error("❌ batchUpdateTransactions error:", err);
    return apiError(res, err.message || "Internal error", "INTERNAL_ERROR", 500);
  }
}
async function batchUpdateWithdrawals(req, res) {
  try {
    console.log("📥 batchUpdateWithdrawals called for user:", req.params.id);
    if (req.user?.role !== ADMIN_ROLE) {
      return apiError(res, "Admin required", "ADMIN_REQUIRED", 403);
    }
    const userId = req.params.id;
    if (!userId) {
      return apiError(res, "User id required", "USER_ID_REQUIRED", 400);
    }
    const BatchWithdrawalsSchema = z$1.object({
      withdrawals: z$1.array(z$1.object({
        id: z$1.string().optional(),
        amount: z$1.number().positive(),
        fee: z$1.number().optional().default(0),
        net_amount: z$1.number().positive(),
        method: z$1.enum(["BANK_TRANSFER", "CRYPTO", "PAYPAL", "WIRE"]).optional().default("BANK_TRANSFER"),
        bank_account: z$1.string().optional().nullable(),
        crypto_address: z$1.string().optional().nullable(),
        status: z$1.enum(["PENDING", "APPROVED", "PROCESSING", "COMPLETED", "REJECTED", "CANCELLED"]).optional().default("PENDING"),
        reference: z$1.string().optional().nullable(),
        notes: z$1.string().optional().nullable(),
        processed_at: z$1.string().optional().nullable(),
        processed_by: z$1.string().optional().nullable()
      }))
    });
    const parsed = BatchWithdrawalsSchema.safeParse(req.body ?? {});
    if (!parsed.success) {
      return apiError(res, "Invalid payload", "VALIDATION_ERROR", 400, parsed.error.errors);
    }
    const { withdrawals } = parsed.data;
    const updates = withdrawals.filter((wd) => wd.id && !wd.id.startsWith("wd_"));
    const inserts = withdrawals.filter((wd) => !wd.id || wd.id.startsWith("wd_"));
    console.log(`🔄 Processing ${updates.length} updates and ${inserts.length} inserts`);
    const updatePromises = updates.map(async (wd) => {
      const updateData = {
        amount: wd.amount,
        fee: wd.fee,
        net_amount: wd.net_amount,
        method: wd.method,
        bank_account: wd.bank_account,
        crypto_address: wd.crypto_address,
        status: wd.status,
        reference: wd.reference,
        notes: wd.notes,
        processed_by: wd.processed_by,
        updated_at: (/* @__PURE__ */ new Date()).toISOString()
      };
      if (wd.status === "COMPLETED" && !wd.processed_at) {
        updateData.processed_at = (/* @__PURE__ */ new Date()).toISOString();
      } else if (wd.processed_at) {
        updateData.processed_at = wd.processed_at;
      }
      return supabase.from("withdrawals").update(updateData).eq("id", wd.id).eq("user_id", userId).select().single();
    });
    const insertPromises = inserts.map(async (wd) => {
      const insertData = {
        user_id: userId,
        amount: wd.amount,
        fee: wd.fee,
        net_amount: wd.net_amount,
        method: wd.method,
        bank_account: wd.bank_account || null,
        crypto_address: wd.crypto_address || null,
        status: wd.status,
        reference: wd.reference || null,
        notes: wd.notes || null,
        processed_at: wd.processed_at || null,
        processed_by: wd.processed_by || null,
        created_at: (/* @__PURE__ */ new Date()).toISOString(),
        updated_at: (/* @__PURE__ */ new Date()).toISOString()
      };
      return supabase.from("withdrawals").insert(insertData).select().single();
    });
    const results = await Promise.allSettled([...updatePromises, ...insertPromises]);
    const successful = results.filter((r) => r.status === "fulfilled").length;
    const failed = results.filter((r) => r.status === "rejected").length;
    results.forEach((result, index) => {
      if (result.status === "rejected") {
        console.error(`Withdrawal operation ${index} failed:`, result.reason);
      }
    });
    console.log(`✅ Batch update completed: ${successful} successful, ${failed} failed`);
    const { data: allWithdrawals, error } = await supabase.from("withdrawals").select("*").eq("user_id", userId).order("created_at", { ascending: false });
    if (error) {
      console.error("Error fetching withdrawals:", error);
      return apiError(res, "Failed to fetch updated withdrawals", "DB_ERROR", 500);
    }
    await writeAdminLog(req.user?.id, "BATCH_UPDATE_WITHDRAWALS", userId, {
      updates: updates.length,
      inserts: inserts.length,
      successful,
      failed
    }, req.ip);
    return apiSuccess(res, { withdrawals: allWithdrawals || [], meta: { successful, failed } });
  } catch (err) {
    console.error("❌ batchUpdateWithdrawals error:", err);
    return apiError(res, err.message || "Internal error", "INTERNAL_ERROR", 500);
  }
}
async function batchUpdateUserPackages(req, res) {
  try {
    console.log("📥 batchUpdateUserPackages called for user:", req.params.id);
    if (req.user?.role !== ADMIN_ROLE) {
      return apiError(res, "Admin required", "ADMIN_REQUIRED", 403);
    }
    const userId = req.params.id;
    if (!userId) {
      return apiError(res, "User id required", "USER_ID_REQUIRED", 400);
    }
    const BatchUserPackagesSchema = z$1.object({
      packages: z$1.array(z$1.object({
        id: z$1.string().optional(),
        package_id: z$1.string(),
        // Required FK
        subscribed_at: z$1.string().optional(),
        expires_at: z$1.string().optional().nullable(),
        is_active: z$1.boolean().optional().default(true)
      }))
    });
    const parsed = BatchUserPackagesSchema.safeParse(req.body ?? {});
    if (!parsed.success) {
      return apiError(res, "Invalid payload", "VALIDATION_ERROR", 400, parsed.error.errors);
    }
    const { packages } = parsed.data;
    const updates = packages.filter((pkg) => pkg.id && !pkg.id.startsWith("pkg_"));
    const inserts = packages.filter((pkg) => !pkg.id || pkg.id.startsWith("pkg_"));
    console.log(`🔄 Processing ${updates.length} updates and ${inserts.length} inserts`);
    const packageIds = [...new Set(packages.map((p) => p.package_id))];
    const { data: existingPackages, error: pkgError } = await supabase.from("packages").select("id").in("id", packageIds);
    if (pkgError) {
      return apiError(res, "Failed to validate packages", "DB_ERROR", 500);
    }
    const validIds = new Set(existingPackages?.map((p) => p.id) || []);
    const invalidIds = packageIds.filter((id) => !validIds.has(id));
    if (invalidIds.length > 0) {
      return apiError(res, `Invalid package IDs: ${invalidIds.join(", ")}`, "INVALID_PACKAGE_IDS", 400);
    }
    const updatePromises = updates.map(async (pkg) => {
      const updateData = {
        package_id: pkg.package_id,
        expires_at: pkg.expires_at,
        is_active: pkg.is_active,
        updated_at: (/* @__PURE__ */ new Date()).toISOString()
      };
      return supabase.from("user_packages").update(updateData).eq("id", pkg.id).eq("user_id", userId).select().single();
    });
    const insertPromises = inserts.map(async (pkg) => {
      const insertData = {
        user_id: userId,
        package_id: pkg.package_id,
        subscribed_at: pkg.subscribed_at || (/* @__PURE__ */ new Date()).toISOString(),
        expires_at: pkg.expires_at || null,
        is_active: pkg.is_active ?? true,
        updated_at: (/* @__PURE__ */ new Date()).toISOString()
      };
      return supabase.from("user_packages").insert(insertData).select().single();
    });
    const results = await Promise.allSettled([...updatePromises, ...insertPromises]);
    const successful = results.filter((r) => r.status === "fulfilled").length;
    const failed = results.filter((r) => r.status === "rejected").length;
    results.forEach((result, index) => {
      if (result.status === "rejected") {
        console.error(`User package operation ${index} failed:`, result.reason);
      }
    });
    console.log(`✅ Batch update completed: ${successful} successful, ${failed} failed`);
    const { data: allPackages, error } = await supabase.from("user_packages").select(`
        *,
        packages:package_id (*)
      `).eq("user_id", userId).order("subscribed_at", { ascending: false });
    if (error) {
      console.error("Error fetching user packages:", error);
      return apiError(res, "Failed to fetch updated user packages", "DB_ERROR", 500);
    }
    await writeAdminLog(req.user?.id, "BATCH_UPDATE_USER_PACKAGES", userId, {
      updates: updates.length,
      inserts: inserts.length,
      successful,
      failed
    }, req.ip);
    return apiSuccess(res, { packages: allPackages || [], meta: { successful, failed } });
  } catch (err) {
    console.error("❌ batchUpdateUserPackages error:", err);
    return apiError(res, err.message || "Internal error", "INTERNAL_ERROR", 500);
  }
}
async function getAllPackages(req, res) {
  try {
    if (req.user?.role !== ADMIN_ROLE) {
      return apiError(res, "Admin required", "ADMIN_REQUIRED", 403);
    }
    const { data: packages, error } = await supabase.from("packages").select("*").order("name", { ascending: true });
    if (error) {
      console.error("Error fetching packages:", error);
      return apiError(res, "Database error", "DB_ERROR", 500);
    }
    return apiSuccess(res, packages || []);
  } catch (err) {
    console.error("getAllPackages error:", err);
    return apiError(res, "Internal error", "INTERNAL_ERROR", 500);
  }
}
async function createPackage(req, res) {
  try {
    if (req.user?.role !== ADMIN_ROLE) {
      return apiError(res, "Admin required", "ADMIN_REQUIRED", 403);
    }
    const CreatePackageSchema = z$1.object({
      name: z$1.string().min(1),
      description: z$1.string().optional().nullable(),
      min_investment: z$1.number().positive(),
      max_investment: z$1.number().positive(),
      expected_return: z$1.number(),
      duration: z$1.number().int().positive(),
      risk_level: z$1.enum(["LOW", "MODERATE", "HIGH"]).optional().default("MODERATE"),
      is_premium: z$1.boolean().optional().default(false),
      category: z$1.string().optional().nullable(),
      features: z$1.array(z$1.string()).optional().default([]),
      is_popular: z$1.boolean().optional().default(false),
      price: z$1.number().optional().nullable()
    });
    const parsed = CreatePackageSchema.safeParse(req.body);
    if (!parsed.success) {
      return apiError(res, "Invalid payload", "VALIDATION_ERROR", 400, parsed.error.errors);
    }
    const { data: newPackage, error } = await supabase.from("packages").insert({
      ...parsed.data,
      created_at: (/* @__PURE__ */ new Date()).toISOString(),
      updated_at: (/* @__PURE__ */ new Date()).toISOString()
    }).select().single();
    if (error) {
      console.error("Error creating package:", error);
      return apiError(res, "Database error", "DB_ERROR", 500);
    }
    await writeAdminLog(req.user?.id, "CREATE_PACKAGE", null, { package: newPackage }, req.ip);
    return apiSuccess(res, newPackage, 201);
  } catch (err) {
    console.error("createPackage error:", err);
    return apiError(res, "Internal error", "INTERNAL_ERROR", 500);
  }
}
async function updatePackage(req, res) {
  try {
    if (req.user?.role !== ADMIN_ROLE) {
      return apiError(res, "Admin required", "ADMIN_REQUIRED", 403);
    }
    const packageId = req.params.id;
    if (!packageId) {
      return apiError(res, "Package id required", "PACKAGE_ID_REQUIRED", 400);
    }
    const UpdatePackageSchema = z$1.object({
      name: z$1.string().min(1).optional(),
      description: z$1.string().optional().nullable(),
      min_investment: z$1.number().positive().optional(),
      max_investment: z$1.number().positive().optional(),
      expected_return: z$1.number().optional(),
      duration: z$1.number().int().positive().optional(),
      risk_level: z$1.enum(["LOW", "MODERATE", "HIGH"]).optional(),
      is_premium: z$1.boolean().optional(),
      category: z$1.string().optional().nullable(),
      features: z$1.array(z$1.string()).optional(),
      is_popular: z$1.boolean().optional(),
      price: z$1.number().optional().nullable()
    });
    const parsed = UpdatePackageSchema.safeParse(req.body);
    if (!parsed.success) {
      return apiError(res, "Invalid payload", "VALIDATION_ERROR", 400, parsed.error.errors);
    }
    const { data: updatedPackage, error } = await supabase.from("packages").update({
      ...parsed.data,
      updated_at: (/* @__PURE__ */ new Date()).toISOString()
    }).eq("id", packageId).select().single();
    if (error) {
      console.error("Error updating package:", error);
      return apiError(res, "Database error", "DB_ERROR", 500);
    }
    if (!updatedPackage) {
      return apiError(res, "Package not found", "PACKAGE_NOT_FOUND", 404);
    }
    await writeAdminLog(req.user?.id, "UPDATE_PACKAGE", null, { packageId, updates: parsed.data }, req.ip);
    return apiSuccess(res, updatedPackage);
  } catch (err) {
    console.error("updatePackage error:", err);
    return apiError(res, "Internal error", "INTERNAL_ERROR", 500);
  }
}
async function deletePackage(req, res) {
  try {
    if (req.user?.role !== ADMIN_ROLE) {
      return apiError(res, "Admin required", "ADMIN_REQUIRED", 403);
    }
    const packageId = req.params.id;
    if (!packageId) {
      return apiError(res, "Package id required", "PACKAGE_ID_REQUIRED", 400);
    }
    const { error } = await supabase.from("packages").delete().eq("id", packageId);
    if (error) {
      console.error("Error deleting package:", error);
      return apiError(res, "Database error", "DB_ERROR", 500);
    }
    await writeAdminLog(req.user?.id, "DELETE_PACKAGE", null, { packageId }, req.ip);
    return apiSuccess(res, { message: "Package deleted successfully" });
  } catch (err) {
    console.error("deletePackage error:", err);
    return apiError(res, "Internal error", "INTERNAL_ERROR", 500);
  }
}
const router = express__default.Router();
router.use(authenticate);
router.get("/packages", getAllPackages);
router.post("/packages", createPackage);
router.put("/packages/:id", updatePackage);
router.delete("/packages/:id", deletePackage);
router.get("/users", getAllUsers);
router.get("/users/export", exportUsersCSV);
router.post("/users", createUser);
router.get("/users/:id", getUserDetails);
router.put("/users/:id", updateUser);
router.delete("/users/:id", deleteUser);
router.post("/users/:id/suspend", suspendUser);
router.post("/users/:id/restore", restoreUser);
router.post("/users/:id/change-role", changeUserRole);
router.put("/users/:id/investments/batch", batchUpdateInvestments);
router.put("/users/:id/transactions/batch", batchUpdateTransactions);
router.put("/users/:id/withdrawals/batch", batchUpdateWithdrawals);
router.put("/users/:id/packages/batch", batchUpdateUserPackages);
router.post("/kyc/approve", approveKyc);
router.post("/kyc/reject", rejectKyc);
router.get("/users/:id/transactions", getUserTransactions);
router.get("/users/:id/deposits", getUserDeposits);
router.get("/users/:id/withdrawals", getUserWithdrawals);
router.get("/users/:id/investments", getUserInvestments);
router.get("/users/:id/packages", getUserPackages);
router.post("/adjust-balance", adjustBalance);
router.get("/logs", getAdminLogs);
function createServer() {
  const app2 = express__default();
  app2.use(
    cors({
      origin: process.env.CORS_ORIGIN || "http://localhost:5173",
      credentials: true
    })
  );
  app2.use(express__default.json({ limit: "10mb" }));
  app2.use(express__default.urlencoded({ extended: true, limit: "10mb" }));
  app2.use((req, res, next) => {
    res.set("Cache-Control", "no-cache");
    next();
  });
  app2.get("/api/ping", (_req, res) => {
    const ping = process.env.PING_MESSAGE ?? "ping";
    res.json({ message: ping });
  });
  app2.use("/api/auth", router$2);
  app2.use("/api/dashboard", router$1);
  app2.use("/api/admin", router);
  return app2;
}
const app = createServer();
const port = process.env.PORT || 3e3;
const __dirname = import.meta.dirname;
const distPath = path.join(__dirname, "../spa");
app.use(express.static(distPath));
app.get("*", (req, res) => {
  if (req.path.startsWith("/api/") || req.path.startsWith("/health")) {
    return res.status(404).json({ error: "API endpoint not found" });
  }
  res.sendFile(path.join(distPath, "index.html"));
});
app.listen(port, () => {
  console.log(`🚀 Fusion Starter server running on port ${port}`);
  console.log(`📱 Frontend: http://localhost:${port}`);
  console.log(`🔧 API: http://localhost:${port}/api`);
});
process.on("SIGTERM", () => {
  console.log("🛑 Received SIGTERM, shutting down gracefully");
  process.exit(0);
});
process.on("SIGINT", () => {
  console.log("🛑 Received SIGINT, shutting down gracefully");
  process.exit(0);
});
//# sourceMappingURL=node-build.mjs.map
